/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.iceberg.catalog;

import com.google.inject.Inject;
import jakarta.annotation.PostConstruct;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Function;
import ru.cedrusdata.catalog.CatalogUtils;
import ru.cedrusdata.catalog.core.filesystem.FileSystemService;
import ru.cedrusdata.catalog.core.filesystem.FileSystemSession;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.iceberg.catalog.IcebergCatalogContext;
import ru.cedrusdata.catalog.iceberg.catalog.IcebergCatalogDetails;
import ru.cedrusdata.catalog.iceberg.io.CatalogIcebergFileIOFactory;
import ru.cedrusdata.catalog.iceberg.table.IcebergTableMetadataCache;
import ru.cedrusdata.catalog.spi.CatalogObjectType;
import ru.cedrusdata.catalog.spi.client.ResultPage;
import ru.cedrusdata.catalog.spi.exception.CatalogBadRequestException;
import ru.cedrusdata.catalog.spi.exception.iceberg.IcebergCatalogAlreadyExistsException;
import ru.cedrusdata.catalog.spi.exception.iceberg.IcebergCatalogDoesNotExistException;
import ru.cedrusdata.catalog.spi.model.IcebergCatalogCreateRequest;
import ru.cedrusdata.catalog.spi.model.IcebergCatalogInfo;
import ru.cedrusdata.catalog.spi.model.IcebergCatalogListResponse;
import ru.cedrusdata.catalog.spi.model.IcebergCatalogUpdateRequest;
import ru.cedrusdata.catalog.store.CatalogStore;

public class IcebergCatalogService {
    private final IcebergTableMetadataCache tableMetadataCache;
    private final FileSystemService fileSystemService;
    private final CatalogStore store;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final Lock lock = new ReentrantLock();
    private final ConcurrentHashMap<String, IcebergCatalogDetails> catalogs = new ConcurrentHashMap();

    @Inject
    public IcebergCatalogService(IcebergTableMetadataCache tableMetadataCache, FileSystemService fileSystemService, CatalogStore store) {
        this.tableMetadataCache = Objects.requireNonNull(tableMetadataCache, "tableMetadataCache");
        this.fileSystemService = Objects.requireNonNull(fileSystemService, "fileSystemService");
        this.store = Objects.requireNonNull(store, "store");
    }

    @PostConstruct
    public void initialize() {
        if (!this.initialized.compareAndSet(false, true)) {
            return;
        }
        this.lock.lock();
        try {
            for (IcebergCatalogDetails info : this.store.listIcebergCatalogDetails()) {
                this.catalogs.put(info.catalogName(), info);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void createIcebergCatalog(IcebergCatalogCreateRequest request) {
        String catalogName = CatalogUtils.normalizeObjectName(CatalogObjectType.ICEBERG_CATALOG, request.getCatalogName());
        String fileSystemName = CatalogUtils.normalizeObjectName(CatalogObjectType.FILE_SYSTEM, request.getFileSystemName());
        UUID fileSystemId = this.fileSystemService.resolveId(fileSystemName);
        String fileSystemLocation = this.fileSystemService.checkLocation(CatalogObjectType.ICEBERG_CATALOG, fileSystemId, request.getFileSystemLocation());
        Map properties = request.getProperties() != null ? request.getProperties() : Map.of();
        this.lock.lock();
        try {
            Optional<UUID> id = this.store.createIcebergCatalogIfNotExists(catalogName, request.getDescription(), fileSystemId, fileSystemName, fileSystemLocation, properties);
            if (id.isEmpty()) {
                throw new IcebergCatalogAlreadyExistsException(catalogName);
            }
            IcebergCatalogDetails info = new IcebergCatalogDetails(id.get(), catalogName, request.getDescription(), fileSystemId, fileSystemLocation, properties);
            this.catalogs.put(catalogName, info);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateIcebergCatalog(String catalogName, IcebergCatalogUpdateRequest request) {
        catalogName = CatalogUtils.normalizeObjectName(CatalogObjectType.ICEBERG_CATALOG, catalogName);
        if (request.getCatalogName() == null && request.getDescription() == null && request.getFileSystemLocation() == null && request.getUpdatedProperties() == null && request.getRemovedProperties() == null) {
            throw new CatalogBadRequestException("Nothing to update");
        }
        this.lock.lock();
        try {
            IcebergCatalogDetails info = this.catalogs.get(catalogName);
            if (info == null) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            String newCatalogName = request.getCatalogName() != null ? CatalogUtils.normalizeObjectName(CatalogObjectType.ICEBERG_CATALOG, request.getCatalogName().getValue()) : catalogName;
            String newDescription = request.getDescription() != null ? request.getDescription().getValue() : info.description();
            String newFileSystemLocation = this.fileSystemService.checkLocation(CatalogObjectType.ICEBERG_CATALOG, info.fileSystemId(), request.getFileSystemLocation() != null ? request.getFileSystemLocation().getValue() : info.fileSystemLocation());
            Map<String, String> newProperties = CatalogUtils.mergeProperties(info.properties(), request.getUpdatedProperties(), request.getRemovedProperties());
            boolean updated = this.store.updateIcebergCatalogIfExists(info.catalogId(), newCatalogName, newDescription, newFileSystemLocation, newProperties);
            if (!updated) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            IcebergCatalogDetails newInfo = new IcebergCatalogDetails(info.catalogId(), newCatalogName, newDescription, info.fileSystemId(), newFileSystemLocation, newProperties);
            this.catalogs.put(newCatalogName, newInfo);
            if (!newCatalogName.equals(catalogName)) {
                this.catalogs.remove(catalogName);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteIcebergCatalog(String catalogName) {
        catalogName = CatalogUtils.normalizeObjectName(CatalogObjectType.ICEBERG_CATALOG, catalogName);
        this.lock.lock();
        try {
            IcebergCatalogDetails info = this.catalogs.get(catalogName);
            if (info == null) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            boolean deleted = this.store.deleteIcebergCatalogIfExists(info.catalogId(), info.catalogName());
            if (!deleted) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            this.catalogs.remove(info.catalogName());
            this.tableMetadataCache.invalidateCatalog(info.catalogId());
        }
        finally {
            this.lock.unlock();
        }
    }

    public IcebergCatalogInfo getIcebergCatalog(String catalogName) {
        IcebergCatalogDetails info = this.catalogs.get(catalogName = CatalogUtils.normalizeObjectName(CatalogObjectType.ICEBERG_CATALOG, catalogName));
        if (info == null) {
            throw new IcebergCatalogDoesNotExistException(catalogName);
        }
        try (FileSystemSession fileSystemSession = this.fileSystemService.createSession(CatalogObjectType.ICEBERG_CATALOG, info.fileSystemId());){
            IcebergCatalogInfo icebergCatalogInfo = new IcebergCatalogInfo(info.catalogName(), info.description(), fileSystemSession.factory().getFileSystemName(), info.fileSystemLocation(), info.properties());
            return icebergCatalogInfo;
        }
    }

    public IcebergCatalogListResponse listIcebergCatalogs(ResultPage page) {
        return this.store.listIcebergCatalogs(page);
    }

    public void execute(AuthenticatedPrincipal currentPrincipal, String catalogName, Consumer<IcebergCatalogContext> action) {
        this.execute(currentPrincipal, catalogName, (IcebergCatalogContext catalog) -> {
            action.accept((IcebergCatalogContext)catalog);
            return null;
        });
    }

    public <T> T execute(AuthenticatedPrincipal currentPrincipal, String catalogName, Function<IcebergCatalogContext, T> action) {
        IcebergCatalogDetails info = this.catalogs.get(catalogName = CatalogUtils.normalizeObjectName(CatalogObjectType.ICEBERG_CATALOG, catalogName));
        if (info == null) {
            throw new IcebergCatalogDoesNotExistException(catalogName);
        }
        try (FileSystemSession session = this.fileSystemService.createSession(CatalogObjectType.ICEBERG_CATALOG, info.fileSystemId());){
            IcebergCatalogContext context = new IcebergCatalogContext(currentPrincipal, info, new CatalogIcebergFileIOFactory(session.factory()));
            T t = action.apply(context);
            return t;
        }
    }
}

