/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.core.principal;

import com.google.common.annotations.VisibleForTesting;
import com.google.inject.Inject;
import io.airlift.log.Logger;
import jakarta.annotation.PostConstruct;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import ru.cedrusdata.catalog.CatalogUtils;
import ru.cedrusdata.catalog.config.CatalogConfig;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.core.principal.AuthenticationContext;
import ru.cedrusdata.catalog.core.principal.JwtCredentials;
import ru.cedrusdata.catalog.core.principal.JwtTokenManager;
import ru.cedrusdata.catalog.core.principal.PrincipalDetails;
import ru.cedrusdata.catalog.core.principal.PrincipalIdAndType;
import ru.cedrusdata.catalog.core.principal.PrincipalType;
import ru.cedrusdata.catalog.core.principal.cache.PrincipalServiceCache;
import ru.cedrusdata.catalog.core.principal.cache.PrincipalServiceCacheEntry;
import ru.cedrusdata.catalog.spi.CatalogObjectType;
import ru.cedrusdata.catalog.spi.client.ResultPage;
import ru.cedrusdata.catalog.spi.exception.CatalogAccessTokenDoesNotExistException;
import ru.cedrusdata.catalog.spi.exception.CatalogAuthenticationException;
import ru.cedrusdata.catalog.spi.exception.CatalogAuthorizationException;
import ru.cedrusdata.catalog.spi.exception.CatalogBadRequestException;
import ru.cedrusdata.catalog.spi.exception.CatalogInternalServerErrorException;
import ru.cedrusdata.catalog.spi.exception.CatalogPrincipalAlreadyExistsException;
import ru.cedrusdata.catalog.spi.exception.CatalogPrincipalDoesNotExistException;
import ru.cedrusdata.catalog.spi.exception.CatalogRoleAlreadyGrantedException;
import ru.cedrusdata.catalog.spi.exception.CatalogRoleDoesNotExistException;
import ru.cedrusdata.catalog.spi.exception.CatalogRoleNotGrantedException;
import ru.cedrusdata.catalog.spi.model.AccessTokenCreateRequest;
import ru.cedrusdata.catalog.spi.model.AccessTokenCreateResponse;
import ru.cedrusdata.catalog.spi.model.AccessTokenCreateTemporaryResponse;
import ru.cedrusdata.catalog.spi.model.AccessTokenListResponse;
import ru.cedrusdata.catalog.spi.model.PrincipalCreateRequest;
import ru.cedrusdata.catalog.spi.model.PrincipalInfo;
import ru.cedrusdata.catalog.spi.model.PrincipalListResponse;
import ru.cedrusdata.catalog.spi.model.PrincipalUpdateRequest;
import ru.cedrusdata.catalog.spi.model.RoleGrantRequest;
import ru.cedrusdata.catalog.spi.model.RoleRevokeRequest;
import ru.cedrusdata.catalog.spi.security.CatalogPrivilege;
import ru.cedrusdata.catalog.store.CatalogStore;

public class PrincipalService {
    private static final Logger logger = Logger.get(PrincipalService.class);
    private static final FileAttribute<?> INITIAL_ACCESS_TOKEN_FILE_ATTRIBUTES = PosixFilePermissions.asFileAttribute(Set.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE));
    public static final String ROLE_ADMIN = "builtin.admin";
    public static final String ROLE_ICEBERG_ADMIN = "builtin.iceberg.admin";
    private static final String BUILTIN_PREFIX = "builtin";
    public static final String CLIENT_ID_PREFIX = "U-";
    private final CatalogStore store;
    private final CatalogConfig config;
    private final JwtTokenManager jwtTokenManager;
    private final int jwtTtlSeconds;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final PrincipalServiceCache principalCache;
    private volatile UUID roleIdAdmin;
    private volatile UUID roleIdIcebergAdmin;

    @Inject
    public PrincipalService(CatalogConfig config, CatalogStore store, JwtTokenManager jwtTokenManager, PrincipalServiceCache principalCache) {
        this.store = Objects.requireNonNull(store, "store");
        this.config = Objects.requireNonNull(config, "config");
        this.principalCache = Objects.requireNonNull(principalCache, "principalCache");
        this.jwtTokenManager = Objects.requireNonNull(jwtTokenManager, "jwtTokenManager");
        this.jwtTtlSeconds = Math.max((int)config.getJwtTokenTtl().roundTo(TimeUnit.SECONDS), 1);
    }

    private static String createAccessTokenId() {
        return CLIENT_ID_PREFIX + UUID.randomUUID().toString().replace("-", "");
    }

    private static String createAccessToken(String accessTokenId) {
        return accessTokenId + ":" + CatalogUtils.generateRandomString(72 - accessTokenId.length() - 1);
    }

    @PostConstruct
    public void initialize() {
        String hashedInitialAdminPassword;
        boolean printAccessToken;
        String initialAdminPrincipal;
        if (!this.initialized.compareAndSet(false, true)) {
            return;
        }
        Map<String, UUID> allRoles = this.store.listRoles();
        this.roleIdAdmin = allRoles.get(ROLE_ADMIN);
        if (this.roleIdAdmin == null) {
            throw new CatalogInternalServerErrorException(String.format("%s role is not available", ROLE_ADMIN));
        }
        this.roleIdIcebergAdmin = allRoles.get(ROLE_ICEBERG_ADMIN);
        if (this.roleIdIcebergAdmin == null) {
            throw new CatalogInternalServerErrorException(String.format("%s role is not available", ROLE_ICEBERG_ADMIN));
        }
        if (this.store.countUserPrincipals() > 0L) {
            return;
        }
        try {
            initialAdminPrincipal = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, this.config.getAdminPrincipal());
        }
        catch (Exception e) {
            throw new CatalogInternalServerErrorException(String.format("Invalid property %s: %s", "admin-principal", e.getMessage()), (Throwable)e);
        }
        if (this.config.getAdminPassword() == null || this.config.getAdminPassword().isEmpty()) {
            printAccessToken = true;
            hashedInitialAdminPassword = CatalogUtils.hashPassword(CatalogUtils.generateRandomString(72));
        } else {
            printAccessToken = false;
            hashedInitialAdminPassword = CatalogUtils.hashPassword(this.config.getAdminPassword());
        }
        Optional<UUID> principalId = this.store.createPrincipalIfNotExists(initialAdminPrincipal, PrincipalType.USER, hashedInitialAdminPassword, Map.of(this.roleIdAdmin, ROLE_ADMIN), true, Map.of());
        if (principalId.isPresent()) {
            logger.info("Created initial admin principal: " + initialAdminPrincipal);
            if (printAccessToken) {
                String accessToken = this.createAccessToken(this.principal(initialAdminPrincipal), new AccessTokenCreateRequest(initialAdminPrincipal, "Initial access token")).getAccessToken();
                this.printInitialAccessToken(accessToken);
            } else {
                logger.info("Skipping initial access token creation because an explicit admin password has been provided");
            }
        }
    }

    private void printInitialAccessToken(String accessToken) {
        Path path = this.config.getAdminAccessTokenFilePath().toPath();
        try {
            Files.deleteIfExists(path);
            Path parentPath = path.getParent();
            if (parentPath != null) {
                Files.createDirectories(parentPath, new FileAttribute[0]);
            }
            Files.createFile(path, INITIAL_ACCESS_TOKEN_FILE_ATTRIBUTES);
            Files.writeString(path, (CharSequence)(accessToken + "\n"), StandardOpenOption.TRUNCATE_EXISTING);
            logger.info("Created initial access token file: " + String.valueOf(path));
        }
        catch (IOException e) {
            logger.warn((Throwable)e, "Failed to create initial access token file, token will be printed to the log: " + e.getMessage());
            logger.info("Created initial access token: " + accessToken);
        }
    }

    @VisibleForTesting
    public String getAdminPrincipal() {
        return this.config.getAdminPrincipal();
    }

    @VisibleForTesting
    public String getAdminPassword() {
        return this.config.getAdminPassword();
    }

    public void createPrincipal(AuthenticatedPrincipal currentPrincipal, PrincipalCreateRequest request) {
        boolean active;
        String hashedPassword;
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.PRINCIPAL_CREATE);
        }
        String normalizedPrincipalName = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, request.getPrincipalName());
        if (PrincipalService.isBuiltin(normalizedPrincipalName)) {
            throw new CatalogBadRequestException(String.format("Principal name cannot start with \"%s\"", BUILTIN_PREFIX));
        }
        String principalType = request.getPrincipalType();
        PrincipalType normalizedPrincipalType = PrincipalType.resolveFromString(principalType).orElseThrow(() -> new CatalogBadRequestException("Unsupported principal type: \"" + principalType + "\""));
        if (normalizedPrincipalType == PrincipalType.USER) {
            String password = request.getPassword() != null ? CatalogUtils.normalizeNotEmpty(request.getPassword(), "Password cannot be empty") : null;
            hashedPassword = password != null ? CatalogUtils.hashPassword(password) : null;
            active = request.getActive() != null ? request.getActive() : true;
        } else {
            if (request.getPassword() != null) {
                throw new CatalogBadRequestException("Role cannot have a password");
            }
            if (request.getActive() != null && !request.getActive().booleanValue()) {
                throw new CatalogBadRequestException("Role cannot be inactive");
            }
            hashedPassword = null;
            active = true;
        }
        this.principalCache.withWriteLock(() -> {
            Optional<UUID> principalId = this.store.createPrincipalIfNotExists(normalizedPrincipalName, normalizedPrincipalType, hashedPassword, Map.of(), active, request.getProperties());
            if (principalId.isEmpty()) {
                throw new CatalogPrincipalAlreadyExistsException(normalizedPrincipalName);
            }
        });
    }

    public void updatePrincipal(AuthenticatedPrincipal currentPrincipal, String principalName, PrincipalUpdateRequest request) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.PRINCIPAL_UPDATE);
        }
        String normalizedPrincipalName = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, principalName);
        if (PrincipalService.isBuiltin(normalizedPrincipalName)) {
            throw new CatalogBadRequestException("Cannot update built-in principal");
        }
        Optional newPrincipalName = request.getPrincipalName() != null ? Optional.of(CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, request.getPrincipalName().getValue())) : Optional.empty();
        Optional newPassword = request.getPassword() != null ? Optional.of(CatalogUtils.normalizeNotEmpty(request.getPassword().getValue(), "Password cannot be empty")) : Optional.empty();
        Optional newActive = request.getActive() != null ? Optional.of(request.getActive().getValue()) : Optional.empty();
        Map updatedProperties = request.getUpdatedProperties();
        Set removedProperties = request.getRemovedProperties();
        if (newPrincipalName.isEmpty() && newPassword.isEmpty() && newActive.isEmpty() && (updatedProperties == null || updatedProperties.isEmpty()) && (removedProperties == null || removedProperties.isEmpty())) {
            throw new CatalogBadRequestException("Nothing to update");
        }
        if (newPrincipalName.isPresent() && PrincipalService.isBuiltin((String)newPrincipalName.get())) {
            throw new CatalogBadRequestException(String.format("Principal name cannot start with \"%s\"", BUILTIN_PREFIX));
        }
        this.principalCache.withWriteLock(() -> {
            Optional<PrincipalDetails> principal = this.store.getPrincipalDetailsByName(normalizedPrincipalName, false);
            if (principal.isEmpty()) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            if (newPassword.isPresent() && principal.get().type() == PrincipalType.ROLE) {
                throw new CatalogBadRequestException("Cannot change password for role");
            }
            if (newActive.isPresent() && principal.get().type() == PrincipalType.ROLE) {
                throw new CatalogBadRequestException("Cannot change active flag for role");
            }
            String resolvedPrincipalName = newPrincipalName.orElse(principal.get().name());
            boolean resolvedActive = newActive.orElse(principal.get().active());
            Map<String, String> resolvedProperties = CatalogUtils.mergeProperties(principal.get().properties(), updatedProperties, removedProperties);
            boolean updated = this.store.updatePrincipalIfExists(principal.get().id(), resolvedPrincipalName, resolvedActive, resolvedProperties, newPassword.map(v -> CatalogUtils.hashPassword(v.trim())));
            if (!updated) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            List<String> accessTokenIds = this.store.listPrincipalAccessTokenIds(principal.get().id());
            this.principalCache.invalidatePrincipalNameAndAccessTokenIds(normalizedPrincipalName, accessTokenIds);
        });
    }

    public void deletePrincipal(AuthenticatedPrincipal currentPrincipal, String principalName) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.PRINCIPAL_DELETE);
        }
        String normalizedPrincipalName = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, principalName);
        if (PrincipalService.isBuiltin(normalizedPrincipalName)) {
            throw new CatalogBadRequestException("Cannot delete built-in principal");
        }
        Optional<UUID> principalId = this.store.getPrincipalIdByName(normalizedPrincipalName);
        if (principalId.isEmpty()) {
            throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
        }
        if (currentPrincipal.id().equals(principalId.get())) {
            throw new CatalogBadRequestException("Cannot delete self");
        }
        this.principalCache.withWriteLock(() -> {
            List<String> accessTokenIds = this.store.listPrincipalAccessTokenIds((UUID)principalId.get());
            boolean deleted = this.store.deletePrincipalIfExists((UUID)principalId.get(), normalizedPrincipalName);
            if (!deleted) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            this.principalCache.invalidatePrincipalNameAndAccessTokenIds(normalizedPrincipalName, accessTokenIds);
        });
    }

    public PrincipalInfo getPrincipal(AuthenticatedPrincipal currentPrincipal, String principalName) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.PRINCIPAL_GET);
        }
        Optional<PrincipalInfo> info = this.store.getPrincipal(principalName = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, principalName));
        if (info.isEmpty()) {
            throw new CatalogPrincipalDoesNotExistException(principalName);
        }
        return info.get();
    }

    public PrincipalListResponse listPrincipals(AuthenticatedPrincipal currentPrincipal, Optional<String> principalType, ResultPage page) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.PRINCIPAL_LIST);
        }
        return this.store.listPrincipals(principalType, page);
    }

    public void grantRole(AuthenticatedPrincipal currentPrincipal, String roleName, RoleGrantRequest request) {
        PrincipalService.authorizeRoleGrantRevoke(currentPrincipal, CatalogPrivilege.ROLE_GRANT);
        String normalizedRoleName = CatalogUtils.normalizeObjectName(CatalogObjectType.ROLE, roleName);
        String normalizedPrincipalName = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, request.getPrincipalName());
        this.principalCache.withWriteLock(() -> {
            Set<String> principalNames = normalizedRoleName.equals(normalizedPrincipalName) ? Set.of(normalizedRoleName) : Set.of(normalizedRoleName, normalizedPrincipalName);
            Map<String, PrincipalIdAndType> principals = this.store.getPrincipalIdAndType(principalNames);
            PrincipalIdAndType role = principals.get(normalizedRoleName);
            if (role == null) {
                throw new CatalogRoleDoesNotExistException(normalizedRoleName);
            }
            PrincipalIdAndType principal = principals.get(normalizedPrincipalName);
            if (principal == null) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            if (role.type() == PrincipalType.USER) {
                throw new CatalogBadRequestException(String.format("\"%s\" references a user", normalizedRoleName));
            }
            if (role.id().equals(principal.id())) {
                throw new CatalogBadRequestException("Role and principal cannot be the same");
            }
            boolean granted = this.store.grantRoleIfNotGranted(principal.id(), normalizedPrincipalName, role.id(), normalizedRoleName);
            if (!granted) {
                throw new CatalogRoleAlreadyGrantedException(normalizedPrincipalName, normalizedRoleName);
            }
            List<String> accessTokenIds = this.store.listPrincipalAccessTokenIds(principal.id());
            this.principalCache.invalidatePrincipalNameAndAccessTokenIds(normalizedPrincipalName, accessTokenIds);
        });
    }

    public void revokeRole(AuthenticatedPrincipal currentPrincipal, String roleName, RoleRevokeRequest request) {
        PrincipalService.authorizeRoleGrantRevoke(currentPrincipal, CatalogPrivilege.ROLE_REVOKE);
        String normalizedRoleName = CatalogUtils.normalizeObjectName(CatalogObjectType.ROLE, roleName);
        String normalizedPrincipalName = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, request.getPrincipalName());
        this.principalCache.withWriteLock(() -> {
            Set<String> principalNames = normalizedRoleName.equals(normalizedPrincipalName) ? Set.of(normalizedRoleName) : Set.of(normalizedRoleName, normalizedPrincipalName);
            Map<String, PrincipalIdAndType> principals = this.store.getPrincipalIdAndType(principalNames);
            PrincipalIdAndType role = principals.get(normalizedRoleName);
            if (role == null) {
                throw new CatalogRoleDoesNotExistException(normalizedRoleName);
            }
            PrincipalIdAndType principal = principals.get(normalizedPrincipalName);
            if (principal == null) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            if (role.type() == PrincipalType.USER) {
                throw new CatalogBadRequestException(String.format("\"%s\" references a user", normalizedRoleName));
            }
            if (role.id().equals(principal.id())) {
                throw new CatalogBadRequestException("Role and principal cannot be the same");
            }
            if (role.id().equals(this.roleIdAdmin) && currentPrincipal.id().equals(principal.id()) && currentPrincipal.admin()) {
                throw new CatalogBadRequestException(String.format("Cannot revoke role \"%s\" from self", ROLE_ADMIN));
            }
            boolean revoked = this.store.revokeRoleIfGranted(principal.id(), role.id());
            if (!revoked) {
                throw new CatalogRoleNotGrantedException(normalizedPrincipalName, normalizedRoleName);
            }
            List<String> accessTokenIds = this.store.listPrincipalAccessTokenIds(principal.id());
            this.principalCache.invalidatePrincipalNameAndAccessTokenIds(normalizedPrincipalName, accessTokenIds);
        });
    }

    private static void authorizeRoleGrantRevoke(AuthenticatedPrincipal currentPrincipal, CatalogPrivilege privilege) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(privilege);
        }
    }

    public AccessTokenCreateTemporaryResponse createTemporaryAccessToken(AuthenticationContext actorContext, String subject) {
        AuthenticationContext impersonatedContext = this.impersonate(actorContext, subject);
        String subjectToken = this.jwtTokenManager.wrapCredentials(new JwtCredentials(impersonatedContext.actor().name(), impersonatedContext.subject().name()), this.jwtTtlSeconds);
        return new AccessTokenCreateTemporaryResponse(subjectToken, Integer.valueOf(this.jwtTtlSeconds));
    }

    public Optional<JwtCredentials> getTemporaryAccessTokenCredentials(String token) {
        return this.jwtTokenManager.unwrapCredentials(token);
    }

    public AccessTokenCreateResponse createAccessToken(AuthenticatedPrincipal currentPrincipal, AccessTokenCreateRequest request) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.ACCESS_TOKEN_CREATE);
        }
        String normalizedPrincipalName = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, request.getPrincipalName());
        PrincipalIdAndType principal = this.store.getPrincipalIdAndType(Set.of(normalizedPrincipalName)).get(normalizedPrincipalName);
        if (principal == null) {
            throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
        }
        if (principal.type() != PrincipalType.USER) {
            throw new CatalogBadRequestException("Cannot create access token for " + principal.type().getCaption());
        }
        String accessTokenId = PrincipalService.createAccessTokenId();
        String accessToken = PrincipalService.createAccessToken(accessTokenId);
        String hashedAccessToken = CatalogUtils.hashPassword(accessToken);
        this.principalCache.withWriteLock(() -> this.store.createAccessToken(principal.id(), normalizedPrincipalName, accessTokenId, hashedAccessToken, request.getDescription()));
        return new AccessTokenCreateResponse(accessTokenId, accessToken);
    }

    public void deleteAccessToken(AuthenticatedPrincipal currentPrincipal, String accessTokenId) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.ACCESS_TOKEN_DELETE);
        }
        String normalizedAccessTokenId = CatalogUtils.normalizeNotEmpty(accessTokenId, String.format("%s cannot be empty", CatalogObjectType.ACCESS_TOKEN.getCapitalLetterCaption()));
        Optional<UUID> principalId = this.store.getPrincipalDetailsByAccessTokenId(normalizedAccessTokenId).map(PrincipalDetails::id);
        if (principalId.isEmpty()) {
            throw new CatalogAccessTokenDoesNotExistException(normalizedAccessTokenId);
        }
        this.principalCache.withWriteLock(() -> {
            boolean deleted = this.store.deleteAccessTokenIfExists(normalizedAccessTokenId);
            if (!deleted) {
                throw new CatalogAccessTokenDoesNotExistException(normalizedAccessTokenId);
            }
            this.principalCache.invalidateAccessTokenId(normalizedAccessTokenId);
        });
    }

    public AccessTokenListResponse listAccessTokens(AuthenticatedPrincipal currentPrincipal, Optional<String> principalName, ResultPage page) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.ACCESS_TOKEN_LIST);
        }
        if (principalName.isPresent()) {
            String normalizePrincipalName = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, principalName.get());
            Optional<UUID> principalId = this.store.getPrincipalIdByName(normalizePrincipalName);
            if (principalId.isEmpty()) {
                return new AccessTokenListResponse(List.of(), null);
            }
            return this.store.listPrincipalAccessTokens(principalId.get(), page);
        }
        return this.store.listAccessTokens(page);
    }

    public Optional<AuthenticatedPrincipal> authenticate(String principalName) {
        return this.authenticate(principalName, Optional.empty());
    }

    public Optional<AuthenticatedPrincipal> authenticate(String principalName, String password) {
        password = CatalogUtils.normalizeNotEmpty(password, "Password cannot be empty");
        return this.authenticate(principalName, Optional.of(password));
    }

    private Optional<AuthenticatedPrincipal> authenticate(String principalName, Optional<String> password) {
        return this.principalCache.withReadLock(() -> {
            Optional<PrincipalServiceCacheEntry> cache;
            String normalizedPrincipalName = CatalogUtils.normalizeObjectName(CatalogObjectType.PRINCIPAL, principalName);
            if (password.isEmpty() && (cache = this.principalCache.getByPrincipalName(normalizedPrincipalName)).isPresent()) {
                return cache.map(this::cachedPrincipalToPrincipal);
            }
            Optional<PrincipalDetails> principalDetails = this.store.getPrincipalDetailsByName(normalizedPrincipalName, true);
            if (principalDetails.isEmpty()) {
                return Optional.empty();
            }
            if (!principalDetails.get().active()) {
                return Optional.empty();
            }
            if (principalDetails.get().type() == PrincipalType.ROLE) {
                return Optional.empty();
            }
            if (password.isPresent() && !CatalogUtils.passwordMatches((String)password.get(), principalDetails.get().hashedPassword())) {
                return Optional.empty();
            }
            AuthenticatedPrincipal principal = this.constructPrincipal(principalDetails.get().id(), principalDetails.get().name(), principalDetails.get().roleIds());
            this.principalCache.putPrincipalName(PrincipalService.principalToCachedPrincipal(principal, null));
            return Optional.of(principal);
        });
    }

    public Optional<AuthenticatedPrincipal> authenticateAccessToken(String accessToken) {
        if (accessToken == null) {
            return Optional.empty();
        }
        String[] parts = accessToken.split(":");
        if (parts.length != 2) {
            return Optional.empty();
        }
        String accessTokenId = parts[0].trim();
        if (accessTokenId.isEmpty()) {
            return Optional.empty();
        }
        return this.principalCache.withReadLock(() -> {
            Optional<PrincipalServiceCacheEntry> cached = this.principalCache.getByAccessTokenId(accessTokenId, accessToken);
            if (cached.isPresent()) {
                return cached.map(this::cachedPrincipalToPrincipal);
            }
            Optional<PrincipalDetails> principalDetails = this.store.getPrincipalDetailsByAccessTokenId(accessTokenId);
            if (principalDetails.isEmpty()) {
                return Optional.empty();
            }
            if (!principalDetails.get().active()) {
                return Optional.empty();
            }
            if (!CatalogUtils.passwordMatches(accessToken, principalDetails.get().hashedPassword())) {
                return Optional.empty();
            }
            AuthenticatedPrincipal principal = this.constructPrincipal(principalDetails.get().id(), principalDetails.get().name(), principalDetails.get().roleIds());
            this.principalCache.putAccessTokenId(accessTokenId, PrincipalService.principalToCachedPrincipal(principal, accessToken));
            return Optional.of(principal);
        });
    }

    public AuthenticationContext impersonate(AuthenticationContext context, String subject) {
        AuthenticatedPrincipal subjectPrincipal = this.authenticate(subject).orElseThrow(CatalogAuthenticationException::new);
        if (context.isImpersonated() && !context.subject().id().equals(subjectPrincipal.id())) {
            throw new CatalogAuthenticationException();
        }
        AuthenticatedPrincipal actorPrincipal = context.actor();
        if (!actorPrincipal.admin() && !actorPrincipal.icebergAdmin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.PRINCIPAL_IMPERSONATE);
        }
        return new AuthenticationContext(actorPrincipal, subjectPrincipal);
    }

    @VisibleForTesting
    public AuthenticatedPrincipal principal(String principalName) {
        Optional<PrincipalDetails> principal = this.store.getPrincipalDetailsByName(principalName, false);
        if (principal.isEmpty()) {
            throw new CatalogPrincipalDoesNotExistException(principalName);
        }
        return this.constructPrincipal(principal.get().id(), principal.get().name(), principal.get().roleIds());
    }

    private AuthenticatedPrincipal constructPrincipal(UUID id, String name, Set<UUID> roles) {
        boolean admin = roles.contains(this.roleIdAdmin);
        boolean icebergAdmin = roles.contains(this.roleIdIcebergAdmin);
        return new AuthenticatedPrincipal(id, name, admin, icebergAdmin);
    }

    private static PrincipalServiceCacheEntry principalToCachedPrincipal(AuthenticatedPrincipal principal, String accessToken) {
        return new PrincipalServiceCacheEntry(principal.id(), principal.name(), principal.admin(), principal.icebergAdmin(), accessToken);
    }

    private AuthenticatedPrincipal cachedPrincipalToPrincipal(PrincipalServiceCacheEntry cached) {
        return new AuthenticatedPrincipal(cached.id(), cached.name(), cached.admin(), cached.icebergAdmin());
    }

    private static boolean isBuiltin(String normalizedPrincipalName) {
        return normalizedPrincipalName.startsWith(BUILTIN_PREFIX);
    }
}

