/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.core.filesystem;

import com.google.inject.Inject;
import io.airlift.log.Logger;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import ru.cedrusdata.catalog.CatalogUtils;
import ru.cedrusdata.catalog.core.filesystem.FaultyCatalogFileSystemFactory;
import ru.cedrusdata.catalog.core.filesystem.FileSystemDetails;
import ru.cedrusdata.catalog.core.filesystem.FileSystemSession;
import ru.cedrusdata.catalog.core.filesystem.UsageAwareCatalogFileSystemFactory;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.plugin.CatalogPluginContextProvider;
import ru.cedrusdata.catalog.plugin.CatalogPluginRegistry;
import ru.cedrusdata.catalog.spi.CatalogObjectType;
import ru.cedrusdata.catalog.spi.client.ResultPage;
import ru.cedrusdata.catalog.spi.exception.CatalogAuthorizationException;
import ru.cedrusdata.catalog.spi.exception.CatalogBadRequestException;
import ru.cedrusdata.catalog.spi.exception.CatalogException;
import ru.cedrusdata.catalog.spi.exception.CatalogFileSystemAlreadyExistsException;
import ru.cedrusdata.catalog.spi.exception.CatalogFileSystemDoesNotExistException;
import ru.cedrusdata.catalog.spi.exception.CatalogInternalServerErrorException;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystem;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystemFactory;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystemProvider;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystemType;
import ru.cedrusdata.catalog.spi.model.FileSystemCheckResponse;
import ru.cedrusdata.catalog.spi.model.FileSystemCreateRequest;
import ru.cedrusdata.catalog.spi.model.FileSystemInfo;
import ru.cedrusdata.catalog.spi.model.FileSystemListResponse;
import ru.cedrusdata.catalog.spi.model.FileSystemUpdateRequest;
import ru.cedrusdata.catalog.spi.security.CatalogPrivilege;
import ru.cedrusdata.catalog.store.CatalogStore;

public class FileSystemService {
    private static final Logger log = Logger.get(FileSystemService.class);
    private final CatalogStore catalogStore;
    private final CatalogPluginRegistry pluginRegistry;
    private final CatalogPluginContextProvider pluginContextProvider;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final Lock lock = new ReentrantLock();
    private final ConcurrentHashMap<UUID, UsageAwareCatalogFileSystemFactory> idToFactory = new ConcurrentHashMap();

    @Inject
    public FileSystemService(CatalogStore catalogStore, CatalogPluginRegistry pluginRegistry, CatalogPluginContextProvider pluginContextProvider) {
        this.catalogStore = Objects.requireNonNull(catalogStore, "catalogStore");
        this.pluginRegistry = Objects.requireNonNull(pluginRegistry, "pluginRegistry");
        this.pluginContextProvider = Objects.requireNonNull(pluginContextProvider, "pluginContextProvider");
    }

    @PostConstruct
    public void initialize() {
        if (!this.initialized.compareAndSet(false, true)) {
            return;
        }
        this.lock.lock();
        try {
            for (FileSystemDetails info : this.catalogStore.fileSystemAllDetails()) {
                this.idToFactory.put(info.id(), this.createFileSystemFactory(info.name(), info.type(), info.properties(), true));
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private UsageAwareCatalogFileSystemFactory createFileSystemFactory(String fileSystemName, CatalogFileSystemType fileSystemType, Map<String, String> properties, boolean startup) {
        try {
            Optional<CatalogFileSystemProvider> provider = this.pluginRegistry.getCatalogFileSystemProvider(fileSystemType);
            if (provider.isEmpty()) {
                throw new CatalogInternalServerErrorException(String.format("No plugin for %s file system \"%s\"", fileSystemType, fileSystemName));
            }
            CatalogFileSystemFactory factory = provider.get().createFileSystemFactory(this.pluginContextProvider.get(), fileSystemName, properties);
            return new UsageAwareCatalogFileSystemFactory(fileSystemName, factory);
        }
        catch (Exception e) {
            Object errorMessage = e instanceof CatalogBadRequestException ? ((CatalogBadRequestException)((Object)e)).getOriginalMessage() : String.format("%s file system \"%s\" failed to start: %s", fileSystemType, fileSystemName, e.getMessage());
            if (startup) {
                errorMessage = (String)errorMessage + ". Please ask administrator to re-create the file system";
                CatalogInternalServerErrorException exception = new CatalogInternalServerErrorException((String)errorMessage, (Throwable)e);
                log.error((Throwable)e, (String)errorMessage);
                return new UsageAwareCatalogFileSystemFactory(fileSystemName, new FaultyCatalogFileSystemFactory((CatalogException)exception));
            }
            throw new CatalogBadRequestException((String)errorMessage, (Throwable)e);
        }
    }

    @PreDestroy
    public void stop() {
        this.lock.lock();
        try {
            for (UsageAwareCatalogFileSystemFactory factory : this.idToFactory.values()) {
                factory.close();
            }
            this.idToFactory.clear();
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void createFileSystem(AuthenticatedPrincipal currentPrincipal, FileSystemCreateRequest request) throws CatalogException {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.FILE_SYSTEM_CREATE);
        }
        String name = CatalogUtils.normalizeObjectName(CatalogObjectType.FILE_SYSTEM, request.getFileSystemName());
        CatalogFileSystemType type = CatalogUtils.resolveFileSystemType(request.getType());
        this.lock.lock();
        try {
            UUID id = UUID.randomUUID();
            UsageAwareCatalogFileSystemFactory factory = this.createFileSystemFactory(name, type, request.getProperties(), false);
            boolean created = this.catalogStore.createFileSystemIfNotExists(id, name, request.getDescription(), type, request.getProperties());
            if (!created) {
                factory.close();
                throw new CatalogFileSystemAlreadyExistsException(name);
            }
            this.idToFactory.put(id, factory);
        }
        finally {
            this.lock.unlock();
        }
    }

    public void updateFileSystem(AuthenticatedPrincipal currentPrincipal, String name, FileSystemUpdateRequest request) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.FILE_SYSTEM_UPDATE);
        }
        name = CatalogUtils.normalizeObjectName(CatalogObjectType.FILE_SYSTEM, name);
        if (request.getFileSystemName() == null && request.getDescription() == null && request.getUpdatedProperties() == null && request.getRemovedProperties() == null) {
            throw new CatalogBadRequestException("Nothing to update");
        }
        UsageAwareCatalogFileSystemFactory newFactory = null;
        this.lock.lock();
        try {
            String finalName = name;
            FileSystemDetails info = this.catalogStore.fileSystemDetails(name).orElseThrow(() -> new CatalogFileSystemDoesNotExistException(finalName));
            String newName = request.getFileSystemName() != null ? CatalogUtils.normalizeObjectName(CatalogObjectType.FILE_SYSTEM, request.getFileSystemName().getValue()) : name;
            String newDescription = request.getDescription() != null ? request.getDescription().getValue() : info.description();
            Map<String, String> newProperties = CatalogUtils.mergeProperties(info.properties(), request.getUpdatedProperties(), request.getRemovedProperties());
            newFactory = this.createFileSystemFactory(newName, info.type(), newProperties, false);
            boolean updated = this.catalogStore.updateFileSystemIfExists(info.id(), newName, newDescription, newProperties);
            if (!updated) {
                throw new CatalogFileSystemDoesNotExistException(name);
            }
            UsageAwareCatalogFileSystemFactory oldFactory = this.idToFactory.put(info.id(), newFactory);
            if (oldFactory != null) {
                oldFactory.close();
            }
        }
        catch (Exception e) {
            if (newFactory != null) {
                newFactory.close();
            }
            throw e;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteFileSystem(AuthenticatedPrincipal currentPrincipal, String name) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.FILE_SYSTEM_DELETE);
        }
        name = CatalogUtils.normalizeObjectName(CatalogObjectType.FILE_SYSTEM, name);
        this.lock.lock();
        try {
            String finalName = name;
            UUID id = this.catalogStore.fileSystemDetails(name).map(FileSystemDetails::id).orElseThrow(() -> new CatalogFileSystemDoesNotExistException(finalName));
            boolean deleted = this.catalogStore.deleteFileSystemIfExists(id, name);
            if (!deleted) {
                throw new CatalogFileSystemDoesNotExistException(name);
            }
            UsageAwareCatalogFileSystemFactory oldFactory = this.idToFactory.remove(id);
            if (oldFactory != null) {
                oldFactory.close();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public FileSystemInfo getFileSystem(AuthenticatedPrincipal currentPrincipal, String name) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.FILE_SYSTEM_GET);
        }
        String finalName = name = CatalogUtils.normalizeObjectName(CatalogObjectType.FILE_SYSTEM, name);
        return this.catalogStore.getFileSystem(name).orElseThrow(() -> new CatalogFileSystemDoesNotExistException(finalName));
    }

    public FileSystemListResponse listFileSystems(AuthenticatedPrincipal currentPrincipal, ResultPage page) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.FILE_SYSTEM_LIST);
        }
        return this.catalogStore.listFileSystems(page);
    }

    public FileSystemCheckResponse check(AuthenticatedPrincipal currentPrincipal, String fileSystemName, String location) {
        if (!currentPrincipal.admin()) {
            throw new CatalogAuthorizationException(CatalogPrivilege.FILE_SYSTEM_CHECK);
        }
        fileSystemName = CatalogUtils.normalizeObjectName(CatalogObjectType.FILE_SYSTEM, fileSystemName);
        location = CatalogUtils.normalizeNotEmpty(location, "File system location cannot be empty");
        UUID fileSystemId = this.resolveId(fileSystemName);
        UsageAwareCatalogFileSystemFactory factory = this.idToFactory.get(fileSystemId);
        if (factory == null) {
            throw new CatalogFileSystemDoesNotExistException(fileSystemName);
        }
        try (FileSystemSession session = new FileSystemSession(factory);){
            Optional<String> locationError = session.factory().isLocationValid(location);
            if (locationError.isPresent()) {
                throw new CatalogBadRequestException(locationError.get());
            }
            CatalogFileSystem fileSystem = session.factory().create(currentPrincipal.name());
            try {
                boolean exists = fileSystem.newInputFile(location).exists();
                FileSystemCheckResponse fileSystemCheckResponse = new FileSystemCheckResponse(Boolean.valueOf(exists));
                return fileSystemCheckResponse;
            }
            catch (Exception e) {
                throw new CatalogInternalServerErrorException(e.getMessage(), (Throwable)e);
            }
        }
    }

    public UUID resolveId(String fileSystemName) {
        fileSystemName = CatalogUtils.normalizeObjectName(CatalogObjectType.FILE_SYSTEM, fileSystemName);
        for (Map.Entry<UUID, UsageAwareCatalogFileSystemFactory> entry : this.idToFactory.entrySet()) {
            if (!fileSystemName.equals(entry.getValue().getFileSystemName())) continue;
            return entry.getKey();
        }
        throw new CatalogFileSystemDoesNotExistException(fileSystemName);
    }

    public FileSystemSession createSession(CatalogObjectType objectType, UUID fileSystemId) {
        UsageAwareCatalogFileSystemFactory factory = this.idToFactory.get(fileSystemId);
        if (factory == null) {
            throw new CatalogInternalServerErrorException(String.format("File system referenced by %s does not exist", objectType.getLowerCaseCaption()));
        }
        return new FileSystemSession(factory);
    }

    public String checkLocation(CatalogObjectType objectType, UUID fileSystemId, String location) {
        location = CatalogUtils.normalizeNotEmpty(location, "File system location cannot be empty");
        try (FileSystemSession session = this.createSession(objectType, fileSystemId);){
            Optional<String> error = session.factory().isLocationValid(location);
            if (error.isPresent()) {
                throw new CatalogBadRequestException(error.get());
            }
        }
        return location;
    }
}

