/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.s3;

import com.google.inject.Inject;
import io.airlift.concurrent.Threads;
import io.opentelemetry.api.OpenTelemetry;
import io.opentelemetry.instrumentation.awssdk.v2_2.AwsSdkTelemetry;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.filesystem.s3.S3Context;
import io.trino.filesystem.s3.S3FileSystemConfig;
import io.trino.filesystem.s3.S3FileSystemStats;
import io.trino.filesystem.s3.S3SecurityMappingFileSystemFactory;
import io.trino.filesystem.s3.S3SecurityMappingProvider;
import io.trino.filesystem.s3.S3SecurityMappingResult;
import jakarta.annotation.PreDestroy;
import java.net.URI;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Function;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.auth.credentials.WebIdentityTokenFileCredentialsProvider;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.http.SdkHttpClient;
import software.amazon.awssdk.http.apache.ApacheHttpClient;
import software.amazon.awssdk.http.apache.ProxyConfiguration;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.S3ClientBuilder;
import software.amazon.awssdk.services.s3.S3Configuration;
import software.amazon.awssdk.services.s3.presigner.S3Presigner;
import software.amazon.awssdk.services.sts.StsClient;
import software.amazon.awssdk.services.sts.StsClientBuilder;
import software.amazon.awssdk.services.sts.auth.StsAssumeRoleCredentialsProvider;

final class S3FileSystemLoader
implements Function<Location, TrinoFileSystemFactory> {
    private final Optional<S3SecurityMappingProvider> mappingProvider;
    private final SdkHttpClient httpClient;
    private final S3ClientFactory clientFactory;
    private final S3Presigner preSigner;
    private final S3Context context;
    private final ExecutorService uploadExecutor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)"s3-upload-%s"));

    @Inject
    public S3FileSystemLoader(S3SecurityMappingProvider mappingProvider, OpenTelemetry openTelemetry, S3FileSystemConfig config, S3FileSystemStats stats) {
        this(Optional.of(mappingProvider), openTelemetry, config, stats);
    }

    S3FileSystemLoader(OpenTelemetry openTelemetry, S3FileSystemConfig config, S3FileSystemStats stats) {
        this(Optional.empty(), openTelemetry, config, stats);
    }

    private S3FileSystemLoader(Optional<S3SecurityMappingProvider> mappingProvider, OpenTelemetry openTelemetry, S3FileSystemConfig config, S3FileSystemStats stats) {
        this.mappingProvider = Objects.requireNonNull(mappingProvider, "mappingProvider is null");
        this.httpClient = S3FileSystemLoader.createHttpClient(config);
        Objects.requireNonNull(stats, "stats is null");
        MetricPublisher metricPublisher = stats.newMetricPublisher();
        this.clientFactory = S3FileSystemLoader.s3ClientFactory(this.httpClient, openTelemetry, config, metricPublisher);
        this.preSigner = S3FileSystemLoader.s3PreSigner(this.httpClient, openTelemetry, config, metricPublisher);
        this.context = new S3Context(Math.toIntExact(config.getStreamingPartSize().toBytes()), config.isRequesterPays(), config.getSseType(), config.getSseKmsKeyId(), Optional.empty(), config.getCannedAcl(), config.isSupportsExclusiveCreate());
    }

    @Override
    public TrinoFileSystemFactory apply(Location location) {
        return new S3SecurityMappingFileSystemFactory(this.mappingProvider.orElseThrow(), this.clientFactory, this.preSigner, this.context, location, this.uploadExecutor);
    }

    @PreDestroy
    public void destroy() {
        try (SdkHttpClient sdkHttpClient = this.httpClient;){
            this.uploadExecutor.shutdownNow();
        }
    }

    S3Client createClient() {
        return this.clientFactory.create(Optional.empty());
    }

    S3Presigner createPreSigner() {
        return this.preSigner;
    }

    S3Context context() {
        return this.context;
    }

    Executor uploadExecutor() {
        return this.uploadExecutor;
    }

    private static S3ClientFactory s3ClientFactory(SdkHttpClient httpClient, OpenTelemetry openTelemetry, S3FileSystemConfig config, MetricPublisher metricPublisher) {
        ClientOverrideConfiguration overrideConfiguration = S3FileSystemLoader.createOverrideConfiguration(openTelemetry, config, metricPublisher);
        Optional<AwsCredentialsProvider> staticCredentialsProvider = S3FileSystemLoader.createStaticCredentialsProvider(config);
        Optional<String> staticRegion = Optional.ofNullable(config.getRegion());
        Optional<String> staticEndpoint = Optional.ofNullable(config.getEndpoint());
        boolean pathStyleAccess = config.isPathStyleAccess();
        boolean useWebIdentityTokenCredentialsProvider = config.isUseWebIdentityTokenCredentialsProvider();
        Optional<String> staticIamRole = Optional.ofNullable(config.getIamRole());
        String staticRoleSessionName = config.getRoleSessionName();
        String externalId = config.getExternalId();
        return mapping -> {
            Optional<AwsCredentialsProvider> credentialsProvider = mapping.flatMap(S3SecurityMappingResult::credentialsProvider).or(() -> staticCredentialsProvider);
            Optional<Region> region = mapping.flatMap(S3SecurityMappingResult::region).or(() -> staticRegion);
            Optional<URI> endpoint = mapping.flatMap(S3SecurityMappingResult::endpoint).or(() -> staticEndpoint);
            Optional iamRole = mapping.flatMap(S3SecurityMappingResult::iamRole).or(() -> staticIamRole);
            String roleSessionName = mapping.flatMap(S3SecurityMappingResult::roleSessionName).orElse(staticRoleSessionName);
            S3ClientBuilder s3 = S3Client.builder();
            s3.overrideConfiguration(overrideConfiguration);
            s3.httpClient(httpClient);
            region.map(Region::of).ifPresent(arg_0 -> ((S3ClientBuilder)s3).region(arg_0));
            endpoint.map(URI::create).ifPresent(arg_0 -> ((S3ClientBuilder)s3).endpointOverride(arg_0));
            s3.forcePathStyle(Boolean.valueOf(pathStyleAccess));
            if (useWebIdentityTokenCredentialsProvider) {
                s3.credentialsProvider((AwsCredentialsProvider)WebIdentityTokenFileCredentialsProvider.builder().asyncCredentialUpdateEnabled(Boolean.valueOf(true)).build());
            } else if (iamRole.isPresent()) {
                s3.credentialsProvider((AwsCredentialsProvider)((StsAssumeRoleCredentialsProvider.Builder)((StsAssumeRoleCredentialsProvider.Builder)StsAssumeRoleCredentialsProvider.builder().refreshRequest(request -> request.roleArn((String)iamRole.get()).roleSessionName(roleSessionName).externalId(externalId)).stsClient(S3FileSystemLoader.createStsClient(config, credentialsProvider))).asyncCredentialUpdateEnabled(Boolean.valueOf(true))).build());
            } else {
                credentialsProvider.ifPresent(arg_0 -> ((S3ClientBuilder)s3).credentialsProvider(arg_0));
            }
            return (S3Client)s3.build();
        };
    }

    private static S3Presigner s3PreSigner(SdkHttpClient httpClient, OpenTelemetry openTelemetry, S3FileSystemConfig config, MetricPublisher metricPublisher) {
        Optional<AwsCredentialsProvider> staticCredentialsProvider = S3FileSystemLoader.createStaticCredentialsProvider(config);
        Optional<String> staticRegion = Optional.ofNullable(config.getRegion());
        Optional<String> staticEndpoint = Optional.ofNullable(config.getEndpoint());
        boolean pathStyleAccess = config.isPathStyleAccess();
        boolean useWebIdentityTokenCredentialsProvider = config.isUseWebIdentityTokenCredentialsProvider();
        Optional<String> staticIamRole = Optional.ofNullable(config.getIamRole());
        String staticRoleSessionName = config.getRoleSessionName();
        String externalId = config.getExternalId();
        S3Presigner.Builder s3 = S3Presigner.builder();
        s3.s3Client(S3FileSystemLoader.s3ClientFactory(httpClient, openTelemetry, config, metricPublisher).create(Optional.empty()));
        staticRegion.map(Region::of).ifPresent(arg_0 -> ((S3Presigner.Builder)s3).region(arg_0));
        staticEndpoint.map(URI::create).ifPresent(arg_0 -> ((S3Presigner.Builder)s3).endpointOverride(arg_0));
        s3.serviceConfiguration((S3Configuration)S3Configuration.builder().pathStyleAccessEnabled(Boolean.valueOf(pathStyleAccess)).build());
        if (useWebIdentityTokenCredentialsProvider) {
            s3.credentialsProvider((AwsCredentialsProvider)WebIdentityTokenFileCredentialsProvider.builder().asyncCredentialUpdateEnabled(Boolean.valueOf(true)).build());
        } else if (staticIamRole.isPresent()) {
            s3.credentialsProvider((AwsCredentialsProvider)((StsAssumeRoleCredentialsProvider.Builder)((StsAssumeRoleCredentialsProvider.Builder)StsAssumeRoleCredentialsProvider.builder().refreshRequest(request -> request.roleArn((String)staticIamRole.get()).roleSessionName(staticRoleSessionName).externalId(externalId)).stsClient(S3FileSystemLoader.createStsClient(config, staticCredentialsProvider))).asyncCredentialUpdateEnabled(Boolean.valueOf(true))).build());
        } else {
            staticCredentialsProvider.ifPresent(arg_0 -> ((S3Presigner.Builder)s3).credentialsProvider(arg_0));
        }
        return s3.build();
    }

    private static Optional<AwsCredentialsProvider> createStaticCredentialsProvider(S3FileSystemConfig config) {
        if (config.getAwsAccessKey() != null || config.getAwsSecretKey() != null) {
            return Optional.of(StaticCredentialsProvider.create((AwsCredentials)AwsBasicCredentials.create((String)config.getAwsAccessKey(), (String)config.getAwsSecretKey())));
        }
        return Optional.empty();
    }

    private static StsClient createStsClient(S3FileSystemConfig config, Optional<AwsCredentialsProvider> credentialsProvider) {
        StsClientBuilder sts = StsClient.builder();
        Optional.ofNullable(config.getStsEndpoint()).map(URI::create).ifPresent(arg_0 -> ((StsClientBuilder)sts).endpointOverride(arg_0));
        Optional.ofNullable(config.getStsRegion()).or(() -> Optional.ofNullable(config.getRegion())).map(Region::of).ifPresent(arg_0 -> ((StsClientBuilder)sts).region(arg_0));
        credentialsProvider.ifPresent(arg_0 -> ((StsClientBuilder)sts).credentialsProvider(arg_0));
        return (StsClient)sts.build();
    }

    private static ClientOverrideConfiguration createOverrideConfiguration(OpenTelemetry openTelemetry, S3FileSystemConfig config, MetricPublisher metricPublisher) {
        return (ClientOverrideConfiguration)ClientOverrideConfiguration.builder().addExecutionInterceptor(AwsSdkTelemetry.builder((OpenTelemetry)openTelemetry).setCaptureExperimentalSpanAttributes(true).setRecordIndividualHttpError(true).build().newExecutionInterceptor()).retryStrategy(S3FileSystemConfig.RetryMode.getRetryStrategy(config.getRetryMode()).toBuilder().maxAttempts(config.getMaxErrorRetries()).build()).addMetricPublisher(metricPublisher).build();
    }

    private static SdkHttpClient createHttpClient(S3FileSystemConfig config) {
        ApacheHttpClient.Builder client = ApacheHttpClient.builder().maxConnections(config.getMaxConnections()).tcpKeepAlive(Boolean.valueOf(config.getTcpKeepAlive()));
        config.getConnectionTtl().ifPresent(ttl -> client.connectionTimeToLive(ttl.toJavaTime()));
        config.getConnectionMaxIdleTime().ifPresent(time -> client.connectionMaxIdleTime(time.toJavaTime()));
        config.getSocketConnectTimeout().ifPresent(timeout -> client.connectionTimeout(timeout.toJavaTime()));
        config.getSocketReadTimeout().ifPresent(timeout -> client.socketTimeout(timeout.toJavaTime()));
        if (config.getHttpProxy() != null) {
            client.proxyConfiguration((ProxyConfiguration)ProxyConfiguration.builder().endpoint(URI.create("%s://%s".formatted(config.isHttpProxySecure() ? "https" : "http", config.getHttpProxy()))).username(config.getHttpProxyUsername()).password(config.getHttpProxyPassword()).nonProxyHosts(config.getNonProxyHosts()).preemptiveBasicAuthenticationEnabled(Boolean.valueOf(config.getHttpProxyPreemptiveBasicProxyAuth())).build());
        }
        return client.build();
    }

    static interface S3ClientFactory {
        public S3Client create(Optional<S3SecurityMappingResult> var1);
    }
}

