/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.core.filesystem;

import com.google.inject.Inject;
import io.airlift.log.Logger;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import ru.cedrusdata.catalog.CatalogObjectNameValidation;
import ru.cedrusdata.catalog.CatalogUtils;
import ru.cedrusdata.catalog.core.AbstractUsageAwareResource;
import ru.cedrusdata.catalog.core.filesystem.FaultyCatalogFileSystemFactory;
import ru.cedrusdata.catalog.core.filesystem.FileSystemDetails;
import ru.cedrusdata.catalog.core.filesystem.FileSystemSession;
import ru.cedrusdata.catalog.core.filesystem.UsageAwareCatalogFileSystemFactory;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.core.security.authorization.Authorizer;
import ru.cedrusdata.catalog.core.security.authorization.PrivilegeInternalService;
import ru.cedrusdata.catalog.core.security.authorization.predicate.AuthorizerFileSystemPredicate;
import ru.cedrusdata.catalog.core.security.authorization.securable.FileSystemInternalSecurable;
import ru.cedrusdata.catalog.core.security.authorization.securable.SecurableItem;
import ru.cedrusdata.catalog.plugin.CatalogPluginContextProvider;
import ru.cedrusdata.catalog.plugin.CatalogPluginRegistry;
import ru.cedrusdata.catalog.plugin.ExtensionPropertyClassifier;
import ru.cedrusdata.catalog.spi.client.ResultPage;
import ru.cedrusdata.catalog.spi.exception.CatalogBadRequestException;
import ru.cedrusdata.catalog.spi.exception.CatalogException;
import ru.cedrusdata.catalog.spi.exception.CatalogFileSystemAlreadyExistsException;
import ru.cedrusdata.catalog.spi.exception.CatalogFileSystemCheckException;
import ru.cedrusdata.catalog.spi.exception.CatalogFileSystemDoesNotExistException;
import ru.cedrusdata.catalog.spi.exception.CatalogInternalServerErrorException;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystem;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystemFactory;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystemProvider;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystemType;
import ru.cedrusdata.catalog.spi.model.FileSystemCheckResponse;
import ru.cedrusdata.catalog.spi.model.FileSystemCreateRequest;
import ru.cedrusdata.catalog.spi.model.FileSystemInfo;
import ru.cedrusdata.catalog.spi.model.FileSystemListResponse;
import ru.cedrusdata.catalog.spi.model.FileSystemUpdateRequest;
import ru.cedrusdata.catalog.store.FileSystemStore;

public class FileSystemService {
    private static final Logger log = Logger.get(FileSystemService.class);
    private final Authorizer authorizer;
    private final FileSystemStore fileSystemStore;
    private final CatalogPluginRegistry pluginRegistry;
    private final CatalogPluginContextProvider pluginContextProvider;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final Lock lock = new ReentrantLock();
    private final ConcurrentHashMap<UUID, UsageAwareCatalogFileSystemFactory> idToFactory = new ConcurrentHashMap();
    private final PrivilegeInternalService privilegeInternalService;
    private volatile ExtensionPropertyClassifier<CatalogFileSystemType> fileSystemPropertyClassifier;

    @Inject
    public FileSystemService(Authorizer authorizer, FileSystemStore fileSystemStore, CatalogPluginRegistry pluginRegistry, CatalogPluginContextProvider pluginContextProvider, PrivilegeInternalService privilegeInternalService) {
        this.authorizer = Objects.requireNonNull(authorizer, "authorizer");
        this.fileSystemStore = Objects.requireNonNull(fileSystemStore, "fileSystemStore");
        this.pluginRegistry = Objects.requireNonNull(pluginRegistry, "pluginRegistry");
        this.pluginContextProvider = Objects.requireNonNull(pluginContextProvider, "pluginContextProvider");
        this.privilegeInternalService = Objects.requireNonNull(privilegeInternalService, "privilegeInternalService");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @PostConstruct
    public void initialize() {
        if (!this.initialized.compareAndSet(false, true)) {
            return;
        }
        ExtensionPropertyClassifier.Builder<CatalogFileSystemType> fileSystemPropertyClassifierBuilder = new ExtensionPropertyClassifier.Builder<CatalogFileSystemType>(CatalogFileSystemType.class);
        for (CatalogFileSystemType type : CatalogFileSystemType.values()) {
            Optional<CatalogFileSystemProvider> provider = this.pluginRegistry.getCatalogFileSystemProvider(type);
            if (!provider.isPresent()) continue;
            fileSystemPropertyClassifierBuilder.add(type, provider.get().redactedPropertyKeys());
        }
        this.fileSystemPropertyClassifier = fileSystemPropertyClassifierBuilder.build();
        this.lock.lock();
        try {
            for (FileSystemDetails info : this.fileSystemStore.fileSystemAllDetails()) {
                this.idToFactory.put(info.id(), this.createFileSystemFactory(info.name(), info.ownerId(), info.type(), info.properties(), true));
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private UsageAwareCatalogFileSystemFactory createFileSystemFactory(String fileSystemName, Optional<UUID> ownerId, CatalogFileSystemType fileSystemType, Map<String, String> properties, boolean startup) {
        try {
            Optional<CatalogFileSystemProvider> provider = this.pluginRegistry.getCatalogFileSystemProvider(fileSystemType);
            if (provider.isEmpty()) {
                throw new CatalogInternalServerErrorException(String.format("No plugin for %s file system \"%s\"", fileSystemType, fileSystemName));
            }
            CatalogFileSystemFactory factory = provider.get().createFileSystemFactory(this.pluginContextProvider.get(), fileSystemName, properties);
            return new UsageAwareCatalogFileSystemFactory(fileSystemName, ownerId, factory);
        }
        catch (Exception e) {
            Object errorMessage = e instanceof CatalogBadRequestException ? ((CatalogBadRequestException)((Object)e)).getOriginalMessage() : String.format("%s file system \"%s\" failed to start: %s", fileSystemType, fileSystemName, e.getMessage());
            if (startup) {
                errorMessage = (String)errorMessage + ". Please ask administrator to re-create the file system";
                CatalogInternalServerErrorException exception = new CatalogInternalServerErrorException((String)errorMessage, (Throwable)e);
                log.error((Throwable)e, (String)errorMessage);
                return new UsageAwareCatalogFileSystemFactory(fileSystemName, ownerId, new FaultyCatalogFileSystemFactory((CatalogException)exception));
            }
            throw new CatalogBadRequestException((String)errorMessage, (Throwable)e);
        }
    }

    @PreDestroy
    public void stop() {
        this.lock.lock();
        try {
            for (UsageAwareCatalogFileSystemFactory factory : this.idToFactory.values()) {
                factory.close();
            }
            this.idToFactory.clear();
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void createFileSystem(AuthenticatedPrincipal currentPrincipal, FileSystemCreateRequest request) throws CatalogException {
        this.authorizer.authorizeFileSystemCreate(currentPrincipal);
        String name = CatalogObjectNameValidation.VALIDATION_FILE_SYSTEM.normalizeObjectName(request.getFileSystemName());
        CatalogFileSystemType type = CatalogUtils.resolveFileSystemType(request.getType());
        this.lock.lock();
        try {
            UUID id = UUID.randomUUID();
            UsageAwareCatalogFileSystemFactory factory = this.createFileSystemFactory(name, Optional.of(currentPrincipal.id()), type, request.getProperties(), false);
            boolean created = this.fileSystemStore.createFileSystemIfNotExists(id, currentPrincipal.id(), name, request.getDescription(), type, request.getProperties());
            if (!created) {
                factory.close();
                throw new CatalogFileSystemAlreadyExistsException(name);
            }
            this.idToFactory.put(id, factory);
        }
        finally {
            this.lock.unlock();
        }
    }

    public void updateFileSystem(AuthenticatedPrincipal currentPrincipal, String fileSystemName, FileSystemUpdateRequest request) {
        fileSystemName = CatalogObjectNameValidation.VALIDATION_FILE_SYSTEM.normalizeObjectName(fileSystemName);
        AbstractUsageAwareResource newFactory = null;
        this.lock.lock();
        try {
            Optional<FileSystemDetails> details = this.fileSystemStore.fileSystemDetails(fileSystemName);
            if (details.isEmpty()) {
                throw new CatalogFileSystemDoesNotExistException(fileSystemName);
            }
            this.authorizer.authorizeFileSystemAlter(currentPrincipal, details.get().toSecurable());
            if (request.getFileSystemName() == null && request.getDescription() == null && request.getUpdatedProperties() == null && request.getRemovedProperties() == null) {
                throw new CatalogBadRequestException("Nothing to update");
            }
            String newName = request.getFileSystemName() != null ? CatalogObjectNameValidation.VALIDATION_FILE_SYSTEM.normalizeObjectName(request.getFileSystemName().getValue()) : fileSystemName;
            String newDescription = request.getDescription() != null ? request.getDescription().getValue() : details.get().description();
            Map<String, String> newProperties = CatalogUtils.mergeProperties(details.get().properties(), request.getUpdatedProperties(), request.getRemovedProperties());
            newFactory = this.createFileSystemFactory(newName, details.get().ownerId(), details.get().type(), newProperties, false);
            boolean updated = this.fileSystemStore.updateFileSystemIfExists(details.get().id(), newName, newDescription, newProperties);
            if (!updated) {
                throw new CatalogFileSystemDoesNotExistException(fileSystemName);
            }
            UsageAwareCatalogFileSystemFactory oldFactory = this.idToFactory.put(details.get().id(), (UsageAwareCatalogFileSystemFactory)newFactory);
            if (oldFactory != null) {
                oldFactory.close();
            }
        }
        catch (Exception e) {
            if (newFactory != null) {
                newFactory.close();
            }
            throw e;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteFileSystem(AuthenticatedPrincipal currentPrincipal, String fileSystemName) {
        fileSystemName = CatalogObjectNameValidation.VALIDATION_FILE_SYSTEM.normalizeObjectName(fileSystemName);
        this.lock.lock();
        try {
            Optional<FileSystemDetails> details = this.fileSystemStore.fileSystemDetails(fileSystemName);
            if (details.isEmpty()) {
                throw new CatalogFileSystemDoesNotExistException(fileSystemName);
            }
            this.authorizer.authorizeFileSystemDrop(currentPrincipal, details.get().toSecurable());
            UUID id = details.get().id();
            boolean deleted = this.fileSystemStore.deleteFileSystemIfExists(id, fileSystemName);
            if (!deleted) {
                throw new CatalogFileSystemDoesNotExistException(fileSystemName);
            }
            UsageAwareCatalogFileSystemFactory oldFactory = this.idToFactory.remove(id);
            if (oldFactory != null) {
                oldFactory.close();
            }
            this.privilegeInternalService.clearPrivileges();
        }
        finally {
            this.lock.unlock();
        }
    }

    public FileSystemInfo info(AuthenticatedPrincipal currentPrincipal, String fileSystemName) {
        Optional<FileSystemDetails> details = this.fileSystemStore.fileSystemDetails(fileSystemName = CatalogObjectNameValidation.VALIDATION_FILE_SYSTEM.normalizeObjectName(fileSystemName));
        if (details.isEmpty()) {
            throw new CatalogFileSystemDoesNotExistException(fileSystemName);
        }
        this.authorizer.authorizeFileSystemDescribe(currentPrincipal, details.get().toSecurable());
        return details.get().toInfo(this.fileSystemPropertyClassifier);
    }

    public FileSystemListResponse listFileSystems(AuthenticatedPrincipal currentPrincipal, ResultPage page) {
        AuthorizerFileSystemPredicate authorizerPredicate = this.authorizer.authorizeFileSystemList(currentPrincipal);
        return this.fileSystemStore.listFileSystems(page, this.fileSystemPropertyClassifier, details -> authorizerPredicate.test(details.toSecurable()));
    }

    public FileSystemCheckResponse check(AuthenticatedPrincipal currentPrincipal, String fileSystemName) {
        String normalizedFileSystemName = CatalogObjectNameValidation.VALIDATION_FILE_SYSTEM.normalizeObjectName(fileSystemName);
        FileSystemInternalSecurable fileSystemSecurable = this.resolveSecurable(normalizedFileSystemName);
        this.authorizer.authorizeFileSystemCheck(currentPrincipal, fileSystemSecurable);
        try (FileSystemSession session = this.createCatalogSession(fileSystemSecurable.fileSystem().id());){
            FileSystemCheckResponse fileSystemCheckResponse;
            block14: {
                CatalogFileSystem ignore = ((CatalogFileSystemFactory)session.resource()).create(currentPrincipal.name(), Map.of());
                try {
                    fileSystemCheckResponse = new FileSystemCheckResponse(Boolean.valueOf(true));
                    if (ignore == null) break block14;
                }
                catch (Throwable throwable) {
                    try {
                        if (ignore != null) {
                            try {
                                ignore.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (Exception e) {
                        throw new CatalogFileSystemCheckException(normalizedFileSystemName, e.toString());
                    }
                }
                ignore.close();
            }
            return fileSystemCheckResponse;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean grantOwnership(AuthenticatedPrincipal currentPrincipal, String fileSystemName, AuthenticatedPrincipal newOwnerPrincipal) {
        fileSystemName = CatalogObjectNameValidation.VALIDATION_FILE_SYSTEM.normalizeObjectName(fileSystemName);
        this.lock.lock();
        try {
            Optional<FileSystemDetails> details = this.fileSystemStore.fileSystemDetails(fileSystemName);
            if (details.isEmpty()) {
                throw new CatalogFileSystemDoesNotExistException(fileSystemName);
            }
            UsageAwareCatalogFileSystemFactory oldFactory = this.idToFactory.get(details.get().id());
            if (oldFactory == null) {
                throw new CatalogFileSystemDoesNotExistException(fileSystemName);
            }
            this.authorizer.authorizeGrantOwnership(currentPrincipal, details.get().toSecurable());
            if (details.get().ownerId().equals(Optional.of(newOwnerPrincipal.id()))) {
                boolean bl = false;
                return bl;
            }
            boolean updated = this.fileSystemStore.updateOwner(details.get().id(), newOwnerPrincipal.id());
            if (!updated) {
                throw new CatalogFileSystemDoesNotExistException(fileSystemName);
            }
            UsageAwareCatalogFileSystemFactory newFactory = this.createFileSystemFactory(oldFactory.name(), Optional.of(newOwnerPrincipal.id()), details.get().type(), details.get().properties(), false);
            this.idToFactory.put(details.get().id(), newFactory);
            oldFactory.close();
            boolean bl = true;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    public FileSystemInternalSecurable resolveSecurable(String fileSystemName) {
        fileSystemName = CatalogObjectNameValidation.VALIDATION_FILE_SYSTEM.normalizeObjectName(fileSystemName);
        for (Map.Entry<UUID, UsageAwareCatalogFileSystemFactory> entry : this.idToFactory.entrySet()) {
            if (!fileSystemName.equals(entry.getValue().name())) continue;
            return new FileSystemInternalSecurable(new SecurableItem(entry.getKey(), entry.getValue().getOwnerId(), fileSystemName));
        }
        throw new CatalogFileSystemDoesNotExistException(fileSystemName);
    }

    public String fileSystemName(UUID fileSystemId) {
        UsageAwareCatalogFileSystemFactory factory = this.idToFactory.get(fileSystemId);
        if (factory == null) {
            throw new CatalogInternalServerErrorException("File system referenced by catalog does not exist");
        }
        return factory.name();
    }

    public FileSystemSession createCatalogSession(UUID fileSystemId) {
        UsageAwareCatalogFileSystemFactory factory = this.idToFactory.get(fileSystemId);
        if (factory == null) {
            throw new CatalogInternalServerErrorException("File system referenced by catalog does not exist");
        }
        return new FileSystemSession(factory);
    }

    public Optional<String> checkCatalogLocation(UUID fileSystemId, String location) {
        if (location == null || location.isEmpty()) {
            return Optional.empty();
        }
        location = CatalogUtils.normalizeNotEmpty(location, "File system location cannot be empty");
        try (FileSystemSession session = this.createCatalogSession(fileSystemId);){
            Optional error = ((CatalogFileSystemFactory)session.resource()).isLocationValid(location);
            if (error.isPresent()) {
                throw new CatalogBadRequestException((String)error.get());
            }
        }
        return Optional.of(location);
    }
}

