/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.core.computeengine;

import com.google.inject.Inject;
import io.airlift.log.Logger;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import java.util.function.Supplier;
import ru.cedrusdata.catalog.CatalogObjectNameValidation;
import ru.cedrusdata.catalog.CatalogUtils;
import ru.cedrusdata.catalog.core.computeengine.ComputeEngineDetails;
import ru.cedrusdata.catalog.core.computeengine.ComputeEngineSession;
import ru.cedrusdata.catalog.core.computeengine.FaultyComputeEngine;
import ru.cedrusdata.catalog.core.computeengine.InternalComputeEngineProvider;
import ru.cedrusdata.catalog.core.computeengine.UsageAwareComputeEngine;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.core.security.authorization.Authorizer;
import ru.cedrusdata.catalog.core.security.authorization.PrivilegeInternalService;
import ru.cedrusdata.catalog.core.security.authorization.predicate.AuthorizerComputeEnginePredicate;
import ru.cedrusdata.catalog.plugin.CatalogPluginContextProvider;
import ru.cedrusdata.catalog.plugin.CatalogPluginRegistry;
import ru.cedrusdata.catalog.plugin.ExtensionPropertyClassifier;
import ru.cedrusdata.catalog.spi.client.ResultPage;
import ru.cedrusdata.catalog.spi.computeengine.CatalogComputeEngine;
import ru.cedrusdata.catalog.spi.computeengine.CatalogComputeEngineProvider;
import ru.cedrusdata.catalog.spi.computeengine.CatalogComputeEngineType;
import ru.cedrusdata.catalog.spi.exception.CatalogBadRequestException;
import ru.cedrusdata.catalog.spi.exception.CatalogComputeEngineAlreadyExistsException;
import ru.cedrusdata.catalog.spi.exception.CatalogComputeEngineCheckException;
import ru.cedrusdata.catalog.spi.exception.CatalogComputeEngineDoesNotExistException;
import ru.cedrusdata.catalog.spi.exception.CatalogException;
import ru.cedrusdata.catalog.spi.exception.CatalogInternalServerErrorException;
import ru.cedrusdata.catalog.spi.model.ComputeEngineCheckResponse;
import ru.cedrusdata.catalog.spi.model.ComputeEngineCreateRequest;
import ru.cedrusdata.catalog.spi.model.ComputeEngineInfo;
import ru.cedrusdata.catalog.spi.model.ComputeEngineListResponse;
import ru.cedrusdata.catalog.spi.model.ComputeEngineOperationMetadata;
import ru.cedrusdata.catalog.spi.model.ComputeEngineUpdateRequest;
import ru.cedrusdata.catalog.store.ComputeEngineStore;

public class ComputeEngineService {
    private static final Logger log = Logger.get(ComputeEngineService.class);
    public static final String INTERNAL_ENGINE_NAME = "local";
    private final Authorizer authorizer;
    private final ComputeEngineStore store;
    private final InternalComputeEngineProvider internalEngineProvider;
    private final CatalogPluginRegistry pluginRegistry;
    private final CatalogPluginContextProvider pluginContextProvider;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final Lock lock = new ReentrantLock();
    private final ConcurrentHashMap<UUID, UsageAwareComputeEngine> enginesById = new ConcurrentHashMap();
    private final PrivilegeInternalService privilegeInternalService;
    private volatile ExtensionPropertyClassifier<CatalogComputeEngineType> enginePropertyClassifier;

    @Inject
    public ComputeEngineService(Authorizer authorizer, ComputeEngineStore store, InternalComputeEngineProvider internalEngineProvider, CatalogPluginRegistry pluginRegistry, CatalogPluginContextProvider pluginContextProvider, PrivilegeInternalService privilegeInternalService) {
        this.authorizer = Objects.requireNonNull(authorizer, "authorizer");
        this.store = Objects.requireNonNull(store, "store");
        this.internalEngineProvider = internalEngineProvider;
        this.pluginRegistry = Objects.requireNonNull(pluginRegistry, "pluginRegistry");
        this.pluginContextProvider = Objects.requireNonNull(pluginContextProvider, "pluginContextProvider");
        this.privilegeInternalService = Objects.requireNonNull(privilegeInternalService, "privilegeInternalService");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @PostConstruct
    public void initialize() {
        if (!this.initialized.compareAndSet(false, true)) {
            return;
        }
        this.pluginRegistry.addCatalogComputeEngineProvider(this.internalEngineProvider);
        ExtensionPropertyClassifier.Builder<CatalogComputeEngineType> enginePropertyClassifierBuilder = new ExtensionPropertyClassifier.Builder<CatalogComputeEngineType>(CatalogComputeEngineType.class);
        for (CatalogComputeEngineType type : CatalogComputeEngineType.values()) {
            Optional<CatalogComputeEngineProvider> provider = this.pluginRegistry.getCatalogComputeEngineProvider(type);
            if (!provider.isPresent()) continue;
            enginePropertyClassifierBuilder.add(type, provider.get().redactedPropertyKeys());
        }
        this.enginePropertyClassifier = enginePropertyClassifierBuilder.build();
        this.lock.lock();
        try {
            for (ComputeEngineDetails details : this.store.engineAllDetails()) {
                UsageAwareComputeEngine engine = this.createEngine(details.id(), details.name(), details.type(), details.properties(), details.ownerId(), true);
                this.enginesById.put(details.id(), engine);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @PreDestroy
    public void stop() {
        this.lock.lock();
        try {
            for (UsageAwareComputeEngine engine : this.enginesById.values()) {
                engine.close();
            }
            this.enginesById.clear();
        }
        finally {
            this.lock.unlock();
        }
    }

    public ComputeEngineCheckResponse checkEngine(AuthenticatedPrincipal principal, String engineName) {
        String normalizedEngineName = CatalogObjectNameValidation.VALIDATION_COMPUTE_ENGINE.normalizeObjectName(engineName);
        try (ComputeEngineSession session = this.createSession(normalizedEngineName);){
            this.authorizer.authorizeComputeEngineCheck(principal, session.toSecurable());
            try {
                ComputeEngineCheckResponse computeEngineCheckResponse = new ComputeEngineCheckResponse(((CatalogComputeEngine)session.resource()).check());
                return computeEngineCheckResponse;
            }
            catch (Exception e) {
                throw new CatalogComputeEngineCheckException(normalizedEngineName, e.toString());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void createEngine(AuthenticatedPrincipal principal, ComputeEngineCreateRequest request) {
        this.authorizer.authorizeComputeEngineCreate(principal);
        String name = CatalogObjectNameValidation.VALIDATION_COMPUTE_ENGINE.normalizeObjectName(request.getComputeEngineName());
        CatalogComputeEngineType type = CatalogUtils.resolveComputeEngineType(request.getType());
        if (type == CatalogComputeEngineType.LOCAL) {
            throw new CatalogBadRequestException(String.format("Engine of type \"%s\" cannot be created", CatalogComputeEngineType.LOCAL.name()));
        }
        UUID id = UUID.randomUUID();
        this.lock.lock();
        try {
            UsageAwareComputeEngine newEngine = this.createEngine(id, name, type, request.getProperties(), Optional.of(principal.id()), false);
            if (!this.updateStoreAndEngineMap(id, newEngine, () -> this.store.createEngineIfNotExists(id, name, request.getDescription(), type, request.getProperties(), principal.id()))) {
                throw new CatalogComputeEngineAlreadyExistsException(name);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateEngine(AuthenticatedPrincipal principal, String engineName, ComputeEngineUpdateRequest request) {
        if (request.getNewEngineName() == null && request.getDescription() == null && request.getUpdatedProperties() == null && request.getRemovedProperties() == null) {
            throw new CatalogBadRequestException("Nothing to update");
        }
        engineName = CatalogObjectNameValidation.VALIDATION_COMPUTE_ENGINE.normalizeObjectName(engineName);
        if (request.getNewEngineName() != null && ComputeEngineService.isInternalEngine(engineName)) {
            throw new CatalogBadRequestException(String.format("Engine name \"%s\" cannot be changed", engineName));
        }
        this.lock.lock();
        try {
            ComputeEngineDetails currentDetails = this.engineDetailsFromStore(engineName);
            this.authorizer.authorizeComputeEngineAlter(principal, currentDetails.toSecurable());
            UUID id = currentDetails.id();
            String newName = request.getNewEngineName() != null ? CatalogObjectNameValidation.VALIDATION_COMPUTE_ENGINE.normalizeObjectName(request.getNewEngineName().getValue()) : engineName;
            String newDescription = request.getDescription() != null ? request.getDescription().getValue() : currentDetails.description();
            Map<String, String> newProperties = CatalogUtils.mergeProperties(currentDetails.properties(), request.getUpdatedProperties(), request.getRemovedProperties());
            UsageAwareComputeEngine newEngine = this.createEngine(id, newName, currentDetails.type(), newProperties, currentDetails.ownerId(), false);
            if (!this.updateStoreAndEngineMap(id, newEngine, () -> this.store.updateEngineIfExists(id, newName, newDescription, newProperties))) {
                throw new CatalogComputeEngineDoesNotExistException(engineName);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean grantOwnership(AuthenticatedPrincipal currentPrincipal, String computeEngineName, AuthenticatedPrincipal newOwnerPrincipal) {
        String normalizedComputeEngineName = CatalogObjectNameValidation.VALIDATION_COMPUTE_ENGINE.normalizeObjectName(computeEngineName);
        this.lock.lock();
        try {
            ComputeEngineDetails details = this.engineDetailsFromStore(normalizedComputeEngineName);
            this.authorizer.authorizeGrantOwnership(currentPrincipal, details.toSecurable());
            if (details.ownerId().equals(Optional.of(newOwnerPrincipal.id()))) {
                boolean bl = false;
                return bl;
            }
            UsageAwareComputeEngine newEngine = this.createEngine(details.id(), details.name(), details.type(), details.properties(), Optional.of(newOwnerPrincipal.id()), false);
            if (!this.updateStoreAndEngineMap(details.id(), newEngine, () -> this.store.updateOwner(details.id(), newOwnerPrincipal.id()))) {
                throw new CatalogComputeEngineDoesNotExistException(normalizedComputeEngineName);
            }
            boolean bl = true;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    public ComputeEngineListResponse listEngines(AuthenticatedPrincipal principal, ResultPage page) {
        AuthorizerComputeEnginePredicate authorizerPredicate = this.authorizer.authorizeComputeEngineList(principal);
        Predicate<ComputeEngineDetails> predicate = details -> authorizerPredicate.test(details.toSecurable());
        return this.store.listEngines(page, this.enginePropertyClassifier, predicate, details -> this.supportedOperations(details.type()));
    }

    public ComputeEngineInfo getEngineInfo(AuthenticatedPrincipal principal, String engineName) {
        ComputeEngineDetails details = this.engineDetailsFromStore(engineName);
        this.authorizer.authorizeComputeEngineDescribe(principal, details.toSecurable());
        return details.info(this.enginePropertyClassifier, this.supportedOperations(details.type()));
    }

    private List<ComputeEngineOperationMetadata> supportedOperations(CatalogComputeEngineType engineType) {
        return this.pluginRegistry.getCatalogComputeEngineProvider(engineType).map(CatalogComputeEngineProvider::supportedOperations).orElse(List.of());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteEngine(AuthenticatedPrincipal principal, String engineName) {
        UsageAwareComputeEngine removedEngine;
        String normalizedName = CatalogObjectNameValidation.VALIDATION_COMPUTE_ENGINE.normalizeObjectName(engineName);
        if (ComputeEngineService.isInternalEngine(normalizedName)) {
            throw new CatalogBadRequestException(String.format("Engine \"%s\" cannot be deleted", normalizedName));
        }
        this.lock.lock();
        try {
            ComputeEngineDetails details = this.engineDetailsFromStore(engineName);
            this.authorizer.authorizeComputeEngineDrop(principal, details.toSecurable());
            if (!this.store.deleteEngineIfExists(details.id())) {
                throw new CatalogComputeEngineDoesNotExistException(normalizedName);
            }
            removedEngine = this.enginesById.remove(details.id());
        }
        finally {
            this.lock.unlock();
        }
        if (removedEngine != null) {
            removedEngine.close();
        }
        this.privilegeInternalService.clearPrivileges();
    }

    public ComputeEngineSession createSession(String engineName) {
        UsageAwareComputeEngine engine = this.engineByName(engineName);
        if (engine == null) {
            throw new CatalogComputeEngineDoesNotExistException(engineName);
        }
        return new ComputeEngineSession(engine);
    }

    public Optional<ComputeEngineSession> createSession(UUID engineId) {
        UsageAwareComputeEngine engine = this.enginesById.get(engineId);
        if (engine == null) {
            return Optional.empty();
        }
        return Optional.of(new ComputeEngineSession(engine));
    }

    private ComputeEngineDetails engineDetailsFromStore(String engineName) {
        return this.store.engineDetails(engineName).orElseThrow(() -> new CatalogComputeEngineDoesNotExistException(engineName));
    }

    private UsageAwareComputeEngine engineByName(String engineName) {
        for (UsageAwareComputeEngine engine : this.enginesById.values()) {
            if (!engineName.equals(engine.name())) continue;
            return engine;
        }
        return null;
    }

    private boolean updateStoreAndEngineMap(UUID id, UsageAwareComputeEngine newEngine, Supplier<Boolean> update) {
        try {
            boolean storeUpdated = update.get();
            if (!storeUpdated) {
                newEngine.close();
            } else {
                UsageAwareComputeEngine old = this.enginesById.put(id, newEngine);
                if (old != null) {
                    old.close();
                }
            }
            return storeUpdated;
        }
        catch (Exception e) {
            newEngine.close();
            throw e;
        }
    }

    private UsageAwareComputeEngine createEngine(UUID engineId, String name, CatalogComputeEngineType type, Map<String, String> properties, Optional<UUID> engineOwnerId, boolean startup) {
        CatalogComputeEngine engine;
        Optional<CatalogComputeEngineProvider> provider = this.pluginRegistry.getCatalogComputeEngineProvider(type);
        if (provider.isEmpty()) {
            throw new CatalogInternalServerErrorException(String.format("No plugin for %s compute engine \"%s\"", type, name));
        }
        try {
            engine = provider.get().createComputeEngine(this.pluginContextProvider.get(), properties);
        }
        catch (Exception e) {
            Object errorMessage = e instanceof CatalogBadRequestException ? ((CatalogBadRequestException)((Object)e)).getOriginalMessage() : String.format("%s engine \"%s\" failed to start: %s", type, name, e.getMessage());
            if (startup) {
                errorMessage = (String)errorMessage + ". Please ask administrator to re-create the compute engine";
                CatalogInternalServerErrorException exception = new CatalogInternalServerErrorException((String)errorMessage, (Throwable)e);
                log.error((Throwable)e, (String)errorMessage);
                engine = new FaultyComputeEngine((CatalogException)exception);
            }
            throw new CatalogBadRequestException((String)errorMessage, (Throwable)e);
        }
        return new UsageAwareComputeEngine(engine, name, engineId, engineOwnerId);
    }

    private static boolean isInternalEngine(String engineName) {
        return INTERNAL_ENGINE_NAME.equals(engineName);
    }
}

