/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.hdfs;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.airlift.stats.TimeStat;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.hdfs.HadoopPaths;
import io.trino.filesystem.hdfs.HdfsFileIterator;
import io.trino.filesystem.hdfs.HdfsInputFile;
import io.trino.filesystem.hdfs.HdfsOutputFile;
import io.trino.hdfs.FileSystemUtils;
import io.trino.hdfs.FileSystemWithBatchDelete;
import io.trino.hdfs.HdfsContext;
import io.trino.hdfs.HdfsEnvironment;
import io.trino.hdfs.TrinoHdfsFileSystemStats;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.time.Instant;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.viewfs.ViewFileSystem;
import org.apache.hadoop.hdfs.DistributedFileSystem;

class HdfsFileSystem
implements TrinoFileSystem {
    private static final Map<String, Boolean> KNOWN_HIERARCHICAL_FILESYSTEMS = ImmutableMap.builder().put((Object)"s3", (Object)false).put((Object)"s3a", (Object)false).put((Object)"s3n", (Object)false).put((Object)"hdfs", (Object)true).put((Object)"ofs", (Object)true).put((Object)"o3fs", (Object)true).buildOrThrow();
    private final HdfsEnvironment environment;
    private final HdfsContext context;
    private final TrinoHdfsFileSystemStats stats;
    private final Map<FileSystem, Boolean> hierarchicalFileSystemCache = new IdentityHashMap<FileSystem, Boolean>();

    public HdfsFileSystem(HdfsEnvironment environment, HdfsContext context, TrinoHdfsFileSystemStats stats) {
        this.environment = Objects.requireNonNull(environment, "environment is null");
        this.context = Objects.requireNonNull(context, "context is null");
        this.stats = Objects.requireNonNull(stats, "stats is null");
    }

    public TrinoInputFile newInputFile(Location location) {
        return new HdfsInputFile(location, null, null, this.environment, this.context, this.stats.getOpenFileCalls());
    }

    public TrinoInputFile newInputFile(Location location, long length) {
        return new HdfsInputFile(location, length, null, this.environment, this.context, this.stats.getOpenFileCalls());
    }

    public TrinoInputFile newInputFile(Location location, long length, Instant lastModified) {
        return new HdfsInputFile(location, length, lastModified, this.environment, this.context, this.stats.getOpenFileCalls());
    }

    public TrinoOutputFile newOutputFile(Location location) {
        return new HdfsOutputFile(location, this.environment, this.context, this.stats.getCreateFileCalls());
    }

    public void deleteFile(Location location) throws IOException {
        location.verifyValidFileLocation();
        this.stats.getDeleteFileCalls().newCall();
        Path file = HadoopPaths.hadoopPath(location);
        FileSystem fileSystem = this.environment.getFileSystem(this.context, file);
        this.environment.doAs(this.context.getIdentity(), () -> {
            Object var5_7;
            block11: {
                TimeStat.BlockTimer ignored = this.stats.getDeleteFileCalls().time();
                try {
                    if (this.hierarchical(fileSystem, location) && !fileSystem.getFileStatus(file).isFile()) {
                        throw new IOException("Location is not a file");
                    }
                    if (!fileSystem.delete(file, false)) {
                        throw new IOException("delete failed");
                    }
                    var5_7 = null;
                    if (ignored == null) break block11;
                }
                catch (Throwable throwable) {
                    try {
                        if (ignored != null) {
                            try {
                                ignored.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (FileNotFoundException ignored2) {
                        return null;
                    }
                    catch (IOException e) {
                        this.stats.getDeleteFileCalls().recordException(e);
                        throw new IOException("Delete file %s failed: %s".formatted(location, e.getMessage()), e);
                    }
                }
                ignored.close();
            }
            return var5_7;
        });
    }

    public void deleteFiles(Collection<Location> locations) throws IOException {
        Map pathsGroupedByDirectory = locations.stream().collect(Collectors.groupingBy(location -> HadoopPaths.hadoopPath(location.parentDirectory()), Collectors.mapping(HadoopPaths::hadoopPath, Collectors.toList())));
        for (Map.Entry directoryWithPaths : pathsGroupedByDirectory.entrySet()) {
            FileSystem rawFileSystem = FileSystemUtils.getRawFileSystem(this.environment.getFileSystem(this.context, directoryWithPaths.getKey()));
            this.environment.doAs(this.context.getIdentity(), () -> {
                if (rawFileSystem instanceof FileSystemWithBatchDelete) {
                    FileSystemWithBatchDelete fileSystemWithBatchDelete = (FileSystemWithBatchDelete)rawFileSystem;
                    fileSystemWithBatchDelete.deleteFiles((Collection)directoryWithPaths.getValue());
                } else {
                    for (Path path : (List)directoryWithPaths.getValue()) {
                        this.stats.getDeleteFileCalls().newCall();
                        try {
                            TimeStat.BlockTimer ignored = this.stats.getDeleteFileCalls().time();
                            try {
                                rawFileSystem.delete(path, false);
                            }
                            finally {
                                if (ignored == null) continue;
                                ignored.close();
                            }
                        }
                        catch (IOException e) {
                            this.stats.getDeleteFileCalls().recordException(e);
                            throw e;
                        }
                    }
                }
                return null;
            });
        }
    }

    public void deleteDirectory(Location location) throws IOException {
        this.stats.getDeleteDirectoryCalls().newCall();
        Path directory = HadoopPaths.hadoopPath(location);
        FileSystem fileSystem = this.environment.getFileSystem(this.context, directory);
        this.environment.doAs(this.context.getIdentity(), () -> {
            try {
                block16: {
                    TimeStat.BlockTimer ignored = this.stats.getDeleteDirectoryCalls().time();
                    if (!location.path().isEmpty()) {
                        if (this.hierarchical(fileSystem, location) && !fileSystem.getFileStatus(directory).isDirectory()) {
                            throw new IOException("Location is not a directory");
                        }
                        if (!fileSystem.delete(directory, true) && fileSystem.exists(directory)) {
                            throw new IOException("delete failed");
                        }
                        Object var5_8 = null;
                        return var5_8;
                    }
                    break block16;
                    finally {
                        if (ignored != null) {
                            ignored.close();
                        }
                    }
                }
                FileStatus[] fileStatusArray = fileSystem.listStatus(directory);
                int n = fileStatusArray.length;
                int n2 = 0;
                while (true) {
                    if (n2 >= n) {
                        fileStatusArray = null;
                        return fileStatusArray;
                    }
                    FileStatus status = fileStatusArray[n2];
                    if (!fileSystem.delete(status.getPath(), true) && fileSystem.exists(status.getPath())) {
                        throw new IOException("delete failed");
                    }
                    ++n2;
                }
            }
            catch (FileNotFoundException e) {
                return null;
            }
            catch (IOException e) {
                this.stats.getDeleteDirectoryCalls().recordException(e);
                throw new IOException("Delete directory %s failed %s".formatted(location, e.getMessage()), e);
            }
        });
    }

    public void renameFile(Location source, Location target) throws IOException {
        source.verifyValidFileLocation();
        target.verifyValidFileLocation();
        this.stats.getRenameFileCalls().newCall();
        Path sourcePath = HadoopPaths.hadoopPath(source);
        Path targetPath = HadoopPaths.hadoopPath(target);
        FileSystem fileSystem = this.environment.getFileSystem(this.context, sourcePath);
        this.environment.doAs(this.context.getIdentity(), () -> {
            Object var7_8;
            block11: {
                TimeStat.BlockTimer ignored = this.stats.getRenameFileCalls().time();
                try {
                    if (!fileSystem.getFileStatus(sourcePath).isFile()) {
                        throw new IOException("Source location is not a file");
                    }
                    if (fileSystem.exists(targetPath)) {
                        throw new IOException("Target location already exists");
                    }
                    if (!fileSystem.rename(sourcePath, targetPath)) {
                        throw new IOException("rename failed");
                    }
                    var7_8 = null;
                    if (ignored == null) break block11;
                }
                catch (Throwable throwable) {
                    try {
                        if (ignored != null) {
                            try {
                                ignored.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        this.stats.getRenameFileCalls().recordException(e);
                        throw new IOException("File rename from %s to %s failed: %s".formatted(source, target, e.getMessage()), e);
                    }
                }
                ignored.close();
            }
            return var7_8;
        });
    }

    public FileIterator listFiles(Location location) throws IOException {
        this.stats.getListFilesCalls().newCall();
        Path directory = HadoopPaths.hadoopPath(location);
        FileSystem fileSystem = this.environment.getFileSystem(this.context, directory);
        return this.environment.doAs(this.context.getIdentity(), () -> {
            HdfsFileIterator hdfsFileIterator;
            block9: {
                TimeStat.BlockTimer ignored = this.stats.getListFilesCalls().time();
                try {
                    hdfsFileIterator = new HdfsFileIterator(location, directory, (RemoteIterator<LocatedFileStatus>)fileSystem.listFiles(directory, true));
                    if (ignored == null) break block9;
                }
                catch (Throwable throwable) {
                    try {
                        if (ignored != null) {
                            try {
                                ignored.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (FileNotFoundException e) {
                        return FileIterator.empty();
                    }
                    catch (IOException e) {
                        this.stats.getListFilesCalls().recordException(e);
                        throw new IOException("List files for %s failed: %s".formatted(location, e.getMessage()), e);
                    }
                }
                ignored.close();
            }
            return hdfsFileIterator;
        });
    }

    public Optional<Boolean> directoryExists(Location location) throws IOException {
        this.stats.getDirectoryExistsCalls().newCall();
        Path directory = HadoopPaths.hadoopPath(location);
        FileSystem fileSystem = this.environment.getFileSystem(this.context, directory);
        if (location.path().isEmpty()) {
            return Optional.of(true);
        }
        return this.environment.doAs(this.context.getIdentity(), () -> {
            try (TimeStat.BlockTimer ignored = this.stats.getDirectoryExistsCalls().time();){
                if (!this.hierarchical(fileSystem, location)) {
                    try {
                        if (fileSystem.listStatusIterator(directory).hasNext()) {
                            Optional<Boolean> optional = Optional.of(true);
                            return optional;
                        }
                    }
                    catch (FileNotFoundException e) {
                        Optional optional = Optional.empty();
                        return optional;
                    }
                    Optional optional = Optional.empty();
                    return optional;
                }
                FileStatus fileStatus = fileSystem.getFileStatus(directory);
                Optional<Boolean> optional = Optional.of(fileStatus.isDirectory());
                return optional;
            }
            catch (FileNotFoundException e) {
                return Optional.of(false);
            }
            catch (IOException e) {
                this.stats.getListFilesCalls().recordException(e);
                throw new IOException("Directory exists check for %s failed: %s".formatted(location, e.getMessage()), e);
            }
        });
    }

    public void createDirectory(Location location) throws IOException {
        this.stats.getCreateDirectoryCalls().newCall();
        Path directory = HadoopPaths.hadoopPath(location);
        FileSystem fileSystem = this.environment.getFileSystem(this.context, directory);
        this.environment.doAs(this.context.getIdentity(), () -> {
            if (!this.hierarchical(fileSystem, location)) {
                return null;
            }
            Optional<FsPermission> permission = this.environment.getNewDirectoryPermissions();
            try (TimeStat.BlockTimer ignored = this.stats.getCreateDirectoryCalls().time();){
                if (!fileSystem.mkdirs(directory, (FsPermission)permission.orElse(null))) {
                    throw new IOException("mkdirs failed");
                }
                if (permission.isPresent()) {
                    fileSystem.setPermission(directory, permission.get());
                }
            }
            catch (IOException e) {
                this.stats.getCreateDirectoryCalls().recordException(e);
                throw new IOException("Create directory %s failed: %s".formatted(location, e.getMessage()), e);
            }
            return null;
        });
    }

    public void renameDirectory(Location source, Location target) throws IOException {
        this.stats.getRenameDirectoryCalls().newCall();
        Path sourcePath = HadoopPaths.hadoopPath(source);
        Path targetPath = HadoopPaths.hadoopPath(target);
        FileSystem fileSystem = this.environment.getFileSystem(this.context, sourcePath);
        this.environment.doAs(this.context.getIdentity(), () -> {
            Object var7_8;
            block12: {
                TimeStat.BlockTimer ignored = this.stats.getRenameDirectoryCalls().time();
                try {
                    if (!this.hierarchical(fileSystem, source)) {
                        throw new IOException("Non-hierarchical file system '%s' does not support directory renames".formatted(fileSystem.getScheme()));
                    }
                    if (!fileSystem.getFileStatus(sourcePath).isDirectory()) {
                        throw new IOException("Source location is not a directory");
                    }
                    if (fileSystem.exists(targetPath)) {
                        throw new IOException("Target location already exists");
                    }
                    if (!fileSystem.rename(sourcePath, targetPath)) {
                        throw new IOException("rename failed");
                    }
                    var7_8 = null;
                    if (ignored == null) break block12;
                }
                catch (Throwable throwable) {
                    try {
                        if (ignored != null) {
                            try {
                                ignored.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        this.stats.getRenameDirectoryCalls().recordException(e);
                        throw new IOException("Directory rename from %s to %s failed: %s".formatted(source, target, e.getMessage()), e);
                    }
                }
                ignored.close();
            }
            return var7_8;
        });
    }

    public Set<Location> listDirectories(Location location) throws IOException {
        this.stats.getListDirectoriesCalls().newCall();
        Path directory = HadoopPaths.hadoopPath(location);
        FileSystem fileSystem = this.environment.getFileSystem(this.context, directory);
        return (Set)this.environment.doAs(this.context.getIdentity(), () -> {
            try (TimeStat.BlockTimer ignored = this.stats.getListDirectoriesCalls().time();){
                FileStatus[] files = fileSystem.listStatus(directory);
                if (files.length == 0) {
                    ImmutableSet immutableSet2 = ImmutableSet.of();
                    return immutableSet2;
                }
                if (files[0].getPath().equals((Object)directory)) {
                    throw new IOException("Location is a file, not a directory: " + String.valueOf(location));
                }
                ImmutableSet immutableSet = (ImmutableSet)Stream.of(files).filter(FileStatus::isDirectory).map(file -> HdfsFileIterator.listedLocation(location, directory, file.getPath())).map(file -> file.appendSuffix("/")).collect(ImmutableSet.toImmutableSet());
                return immutableSet;
            }
            catch (FileNotFoundException e) {
                return ImmutableSet.of();
            }
            catch (IOException e) {
                this.stats.getListDirectoriesCalls().recordException(e);
                throw new IOException("List directories for %s failed: %s".formatted(location, e.getMessage()), e);
            }
        });
    }

    public Optional<Location> createTemporaryDirectory(Location targetLocation, String temporaryPrefix, String relativePrefix) throws IOException {
        Preconditions.checkArgument((!relativePrefix.contains("/") ? 1 : 0) != 0, (Object)"relativePrefix must not contain slash");
        this.stats.getCreateTemporaryDirectoryCalls().newCall();
        Path targetPath = HadoopPaths.hadoopPath(targetLocation);
        FileSystem fileSystem = this.environment.getFileSystem(this.context, targetPath);
        return this.environment.doAs(this.context.getIdentity(), () -> {
            try (TimeStat.BlockTimer ignored = this.stats.getCreateTemporaryDirectoryCalls().time();){
                DistributedFileSystem distributedFileSystem;
                FileSystem rawFileSystem = FileSystemUtils.getRawFileSystem(fileSystem);
                String prefix = rawFileSystem instanceof ViewFileSystem ? relativePrefix : temporaryPrefix;
                Path temporaryRoot = new Path(targetPath, prefix);
                Path temporaryPath = new Path(temporaryRoot, UUID.randomUUID().toString());
                Location temporaryLocation = Location.of((String)temporaryPath.toString());
                if (!this.hierarchical(fileSystem, temporaryLocation)) {
                    Optional optional = Optional.empty();
                    return optional;
                }
                if (rawFileSystem instanceof DistributedFileSystem && (distributedFileSystem = (DistributedFileSystem)rawFileSystem).getEZForPath(targetPath) != null) {
                    Optional optional = Optional.empty();
                    return optional;
                }
                Optional<FsPermission> permission = this.environment.getNewDirectoryPermissions();
                if (!fileSystem.mkdirs(temporaryPath, (FsPermission)permission.orElse(null))) {
                    throw new IOException("mkdirs failed for " + String.valueOf(temporaryPath));
                }
                if (permission.isPresent()) {
                    fileSystem.setPermission(temporaryPath, permission.get());
                }
                Optional<Location> optional = Optional.of(temporaryLocation);
                return optional;
            }
            catch (IOException e) {
                this.stats.getCreateTemporaryDirectoryCalls().recordException(e);
                throw new IOException("Create temporary directory for %s failed: %s".formatted(targetLocation, e.getMessage()), e);
            }
        });
    }

    private boolean hierarchical(FileSystem fileSystem, Location rootLocation) {
        Boolean knownResult = KNOWN_HIERARCHICAL_FILESYSTEMS.get(fileSystem.getScheme());
        if (knownResult != null) {
            return knownResult;
        }
        Boolean cachedResult = this.hierarchicalFileSystemCache.get(fileSystem);
        if (cachedResult != null) {
            return cachedResult;
        }
        try {
            fileSystem.listStatus(HadoopPaths.hadoopPath(rootLocation.appendPath(UUID.randomUUID().toString())));
            this.hierarchicalFileSystemCache.putIfAbsent(fileSystem, false);
            return false;
        }
        catch (IOException e) {
            this.hierarchicalFileSystemCache.putIfAbsent(fileSystem, true);
            return true;
        }
    }

    static <T extends Throwable> T withCause(T throwable, Throwable cause) {
        throwable.initCause(cause);
        return throwable;
    }
}

