/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.iceberg.catalog;

import com.google.inject.Inject;
import jakarta.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiFunction;
import java.util.function.Function;
import org.apache.iceberg.rest.responses.ConfigResponse;
import ru.cedrusdata.catalog.CatalogObjectNameValidation;
import ru.cedrusdata.catalog.CatalogUtils;
import ru.cedrusdata.catalog.core.filesystem.FileSystemService;
import ru.cedrusdata.catalog.core.filesystem.FileSystemSession;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.core.security.authorization.Authorizer;
import ru.cedrusdata.catalog.core.security.authorization.PrivilegeInternalService;
import ru.cedrusdata.catalog.core.security.authorization.predicate.AuthorizerCatalogPredicate;
import ru.cedrusdata.catalog.core.security.authorization.securable.FileSystemInternalSecurable;
import ru.cedrusdata.catalog.iceberg.catalog.IcebergCatalogContext;
import ru.cedrusdata.catalog.iceberg.catalog.IcebergCatalogDefaultLocationProvider;
import ru.cedrusdata.catalog.iceberg.catalog.IcebergCatalogDetails;
import ru.cedrusdata.catalog.iceberg.catalog.IcebergCatalogInfoEx;
import ru.cedrusdata.catalog.iceberg.io.CatalogIcebergFileIOFactory;
import ru.cedrusdata.catalog.iceberg.rest.IcebergOptions;
import ru.cedrusdata.catalog.iceberg.table.IcebergTableMetadataCache;
import ru.cedrusdata.catalog.spi.client.ResultPage;
import ru.cedrusdata.catalog.spi.exception.CatalogBadRequestException;
import ru.cedrusdata.catalog.spi.exception.iceberg.IcebergCatalogAlreadyExistsException;
import ru.cedrusdata.catalog.spi.exception.iceberg.IcebergCatalogCheckException;
import ru.cedrusdata.catalog.spi.exception.iceberg.IcebergCatalogDoesNotExistException;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystem;
import ru.cedrusdata.catalog.spi.filesystem.CatalogFileSystemFactory;
import ru.cedrusdata.catalog.spi.model.IcebergCatalogCheckResponse;
import ru.cedrusdata.catalog.spi.model.IcebergCatalogCreateRequest;
import ru.cedrusdata.catalog.spi.model.IcebergCatalogInfo;
import ru.cedrusdata.catalog.spi.model.IcebergCatalogListResponse;
import ru.cedrusdata.catalog.spi.model.IcebergCatalogUpdateRequest;
import ru.cedrusdata.catalog.store.IcebergCatalogStore;

public class IcebergCatalogService {
    private final IcebergTableMetadataCache tableMetadataCache;
    private final FileSystemService fileSystemService;
    private final IcebergCatalogStore store;
    private final Authorizer authorizer;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final Lock lock = new ReentrantLock();
    private final ConcurrentHashMap<String, IcebergCatalogDetails> catalogs = new ConcurrentHashMap();
    private final PrivilegeInternalService privilegeInternalService;

    @Inject
    public IcebergCatalogService(IcebergTableMetadataCache tableMetadataCache, FileSystemService fileSystemService, IcebergCatalogStore store, Authorizer authorizer, PrivilegeInternalService privilegeInternalService) {
        this.tableMetadataCache = Objects.requireNonNull(tableMetadataCache, "tableMetadataCache");
        this.fileSystemService = Objects.requireNonNull(fileSystemService, "fileSystemService");
        this.store = Objects.requireNonNull(store, "store");
        this.authorizer = Objects.requireNonNull(authorizer, "authorizer");
        this.privilegeInternalService = Objects.requireNonNull(privilegeInternalService, "privilegeInternalService");
    }

    @PostConstruct
    public void initialize() {
        if (!this.initialized.compareAndSet(false, true)) {
            return;
        }
        this.lock.lock();
        try {
            for (IcebergCatalogDetails info : this.store.listDetails()) {
                this.catalogs.put(info.catalogName(), info);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void create(AuthenticatedPrincipal principal, IcebergCatalogCreateRequest request) {
        String catalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(request.getCatalogName());
        String fileSystemName = CatalogObjectNameValidation.VALIDATION_FILE_SYSTEM.normalizeObjectName(request.getFileSystemName());
        FileSystemInternalSecurable fileSystemSecurable = this.fileSystemService.resolveSecurable(fileSystemName);
        UUID fileSystemId = fileSystemSecurable.fileSystem().id();
        Optional<String> fileSystemLocation = this.fileSystemService.checkCatalogLocation(fileSystemId, request.getFileSystemLocation());
        Map properties = request.getProperties() != null ? request.getProperties() : Map.of();
        this.authorizer.authorizeCatalogCreate(principal);
        this.lock.lock();
        try {
            Optional<UUID> id = this.store.createIfNotExists(catalogName, request.getDescription(), principal.id(), fileSystemId, fileSystemName, fileSystemLocation, properties);
            if (id.isEmpty()) {
                throw new IcebergCatalogAlreadyExistsException(catalogName);
            }
            IcebergCatalogDetails info = new IcebergCatalogDetails(id.get(), catalogName, Optional.of(principal.id()), request.getDescription(), fileSystemId, fileSystemLocation, properties);
            this.catalogs.put(catalogName, info);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void update(AuthenticatedPrincipal principal, String catalogName, IcebergCatalogUpdateRequest request) {
        catalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(catalogName);
        if (request.getCatalogName() == null && request.getDescription() == null && request.getFileSystemLocation() == null && request.getUpdatedProperties() == null && request.getRemovedProperties() == null) {
            throw new CatalogBadRequestException("Nothing to update");
        }
        this.lock.lock();
        try {
            IcebergCatalogDetails details = this.catalogs.get(catalogName);
            if (details == null) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            this.authorizer.authorizeCatalogAlter(principal, details.toSecurable());
            String newCatalogName = request.getCatalogName() != null ? CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(request.getCatalogName().getValue()) : catalogName;
            String newDescription = request.getDescription() != null ? request.getDescription().getValue() : details.description();
            Optional<String> newFileSystemLocation = details.fileSystemLocation();
            if (request.getFileSystemLocation() != null) {
                newFileSystemLocation = this.fileSystemService.checkCatalogLocation(details.fileSystemId(), request.getFileSystemLocation().getValue());
            }
            Map<String, String> newProperties = CatalogUtils.mergeProperties(details.properties(), request.getUpdatedProperties(), request.getRemovedProperties());
            boolean updated = this.store.updateIfExists(details.catalogId(), newCatalogName, newDescription, newFileSystemLocation, newProperties);
            if (!updated) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            IcebergCatalogDetails newDetails = new IcebergCatalogDetails(details.catalogId(), newCatalogName, details.ownerId(), newDescription, details.fileSystemId(), newFileSystemLocation, newProperties);
            this.catalogs.put(newCatalogName, newDetails);
            if (!newCatalogName.equals(catalogName)) {
                this.catalogs.remove(catalogName);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void delete(AuthenticatedPrincipal principal, String catalogName) {
        catalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(catalogName);
        this.lock.lock();
        try {
            IcebergCatalogDetails details = this.catalogs.get(catalogName);
            if (details == null) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            this.authorizer.authorizeCatalogDrop(principal, details.toSecurable());
            boolean deleted = this.store.deleteIfExists(details.catalogId(), details.catalogName());
            if (!deleted) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            this.catalogs.remove(details.catalogName());
            this.tableMetadataCache.invalidateCatalog(details.catalogId());
            this.privilegeInternalService.clearPrivileges();
        }
        finally {
            this.lock.unlock();
        }
    }

    public IcebergCatalogDetails getDetails(AuthenticatedPrincipal principal, String catalogName) {
        IcebergCatalogDetails details = this.catalogs.get(catalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(catalogName));
        if (details == null) {
            throw new IcebergCatalogDoesNotExistException(catalogName);
        }
        this.authorizer.authorizeCatalogDescribe(principal, details.toSecurable());
        return details;
    }

    public IcebergCatalogInfo info(AuthenticatedPrincipal principal, String catalogName) {
        Optional<IcebergCatalogInfoEx> info = this.store.getInfo(catalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(catalogName));
        if (info.isEmpty()) {
            throw new IcebergCatalogDoesNotExistException(catalogName);
        }
        this.authorizer.authorizeCatalogDescribe(principal, info.get().toSecurable());
        return info.get().info();
    }

    public IcebergCatalogListResponse listInfo(AuthenticatedPrincipal principal, ResultPage page) {
        AuthorizerCatalogPredicate authorizerPredicate = this.authorizer.authorizeCatalogList(principal);
        return this.store.listInfo(page, (IcebergCatalogInfoEx details) -> authorizerPredicate.test(details.toSecurable()));
    }

    public IcebergCatalogCheckResponse check(AuthenticatedPrincipal principal, String catalogName) {
        String normalizedCatalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(catalogName);
        return this.executeWithIo(principal, normalizedCatalogName, (IcebergCatalogContext catalogContext, CatalogIcebergFileIOFactory ioFactory) -> {
            this.authorizer.authorizeCatalogCheck(principal, catalogContext.toSecurable());
            Optional<String> defaultLocation = catalogContext.defaultLocationProvider().defaultLocation();
            if (defaultLocation.isPresent()) {
                try {
                    CatalogFileSystem fileSystem = ioFactory.createFileIO(principal, IcebergOptions.DEFAULT).fileSystem();
                    Optional exists = fileSystem.directoryExists(defaultLocation.get());
                    if (exists.isPresent() && !((Boolean)exists.get()).booleanValue()) {
                        throw new IcebergCatalogCheckException(normalizedCatalogName, String.format("Default location \"%s\" doesn't exist or is not a directory", defaultLocation.get()));
                    }
                }
                catch (IcebergCatalogCheckException e) {
                    throw e;
                }
                catch (Exception e) {
                    throw new IcebergCatalogCheckException(normalizedCatalogName, e.toString());
                }
            }
            return new IcebergCatalogCheckResponse(Boolean.valueOf(true));
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean grantOwnership(AuthenticatedPrincipal principal, String catalogName, AuthenticatedPrincipal newOwnerPrincipal) {
        catalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(catalogName);
        this.lock.lock();
        try {
            IcebergCatalogDetails details = this.catalogs.get(catalogName);
            if (details == null) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            this.authorizer.authorizeGrantOwnership(principal, details.toSecurable());
            if (details.ownerId().equals(Optional.of(newOwnerPrincipal.id()))) {
                boolean bl = false;
                return bl;
            }
            boolean updated = this.store.updateOwnerIfExists(details.catalogId(), newOwnerPrincipal.id());
            if (!updated) {
                throw new IcebergCatalogDoesNotExistException(catalogName);
            }
            IcebergCatalogDetails newDetails = new IcebergCatalogDetails(details.catalogId(), details.catalogName(), Optional.of(newOwnerPrincipal.id()), details.description(), details.fileSystemId(), details.fileSystemLocation(), details.properties());
            this.catalogs.put(details.catalogName(), newDetails);
            boolean bl = true;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    public ConfigResponse getRestConfig(AuthenticatedPrincipal principal, String catalogName) {
        IcebergCatalogDetails details = this.getDetails(principal, catalogName);
        HashMap<String, String> defaults = new HashMap<String, String>();
        if (details.properties() != null) {
            defaults.putAll(details.properties());
        }
        HashMap<String, String> overrides = new HashMap<String, String>();
        overrides.put("rest-metrics-reporting-enabled", Boolean.FALSE.toString());
        overrides.put("prefix", details.catalogName());
        if (details.fileSystemLocation().isPresent()) {
            overrides.put("location", details.fileSystemLocation().get());
        }
        overrides.put("view-endpoints-supported", "true");
        return ConfigResponse.builder().withDefaults(defaults).withOverrides(overrides).build();
    }

    public <T> T execute(AuthenticatedPrincipal principal, String catalogName, Function<IcebergCatalogContext, T> action) {
        IcebergCatalogDetails details = this.catalogs.get(catalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(catalogName));
        if (details == null) {
            throw new IcebergCatalogDoesNotExistException(catalogName);
        }
        return this.execute(principal, details, action);
    }

    public <T> T execute(AuthenticatedPrincipal principal, UUID catalogId, String catalogName, Function<IcebergCatalogContext, T> action) {
        IcebergCatalogDetails details = this.catalogs.get(catalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(catalogName));
        if (details == null || !catalogId.equals(details.catalogId())) {
            throw new IcebergCatalogDoesNotExistException(catalogName);
        }
        return this.execute(principal, details, action);
    }

    public <T> T executeWithIo(AuthenticatedPrincipal principal, String catalogName, BiFunction<IcebergCatalogContext, CatalogIcebergFileIOFactory, T> action) {
        IcebergCatalogDetails details = this.catalogs.get(catalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeObjectName(catalogName));
        if (details == null) {
            throw new IcebergCatalogDoesNotExistException(catalogName);
        }
        return this.executeWithIo(principal, details, action);
    }

    public <T> T executeWithIo(AuthenticatedPrincipal principal, IcebergCatalogDetails details, BiFunction<IcebergCatalogContext, CatalogIcebergFileIOFactory, T> action) {
        return (T)this.execute(principal, details, (IcebergCatalogContext catalogContext) -> {
            try (FileSystemSession fileSystemContext = this.fileSystemService.createCatalogSession(details.fileSystemId());){
                CatalogIcebergFileIOFactory ioFactory = new CatalogIcebergFileIOFactory((CatalogFileSystemFactory)fileSystemContext.resource());
                try {
                    Object r = action.apply((IcebergCatalogContext)catalogContext, ioFactory);
                    ioFactory.close();
                    return r;
                }
                catch (Throwable throwable) {
                    try {
                        ioFactory.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
            }
        });
    }

    private <T> T execute(AuthenticatedPrincipal principal, IcebergCatalogDetails details, Function<IcebergCatalogContext, T> action) {
        IcebergCatalogContext catalogContext = new IcebergCatalogContext(principal, details.catalogId(), details.catalogName(), details.ownerId(), new IcebergCatalogDefaultLocationProvider(details.catalogName(), details.fileSystemLocation()), details.properties().containsKey("cedrusdata-maintenance-principal"));
        return action.apply(catalogContext);
    }
}

