/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.core.principal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.base.Verify;
import com.google.inject.Inject;
import io.airlift.log.Logger;
import jakarta.annotation.PostConstruct;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import ru.cedrusdata.catalog.CatalogObjectNameValidation;
import ru.cedrusdata.catalog.CatalogUtils;
import ru.cedrusdata.catalog.config.CatalogConfig;
import ru.cedrusdata.catalog.core.PageData;
import ru.cedrusdata.catalog.core.principal.AccessTokenInfoEx;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.core.principal.AuthenticationContext;
import ru.cedrusdata.catalog.core.principal.JwtCredentials;
import ru.cedrusdata.catalog.core.principal.JwtTokenManager;
import ru.cedrusdata.catalog.core.principal.PrincipalDetails;
import ru.cedrusdata.catalog.core.principal.PrincipalInfoEx;
import ru.cedrusdata.catalog.core.principal.PrincipalRoleIdService;
import ru.cedrusdata.catalog.core.principal.PrincipalType;
import ru.cedrusdata.catalog.core.principal.cache.PrincipalServiceCache;
import ru.cedrusdata.catalog.core.principal.cache.PrincipalServiceCacheEntry;
import ru.cedrusdata.catalog.core.security.authorization.Authorizer;
import ru.cedrusdata.catalog.core.security.authorization.PrivilegeInternalService;
import ru.cedrusdata.catalog.core.security.authorization.predicate.AuthorizerPrincipalPredicate;
import ru.cedrusdata.catalog.core.security.provider.SecurityProviderService;
import ru.cedrusdata.catalog.spi.client.ResultPage;
import ru.cedrusdata.catalog.spi.exception.CatalogAccessTokenDoesNotExistException;
import ru.cedrusdata.catalog.spi.exception.CatalogAuthenticationException;
import ru.cedrusdata.catalog.spi.exception.CatalogBadRequestException;
import ru.cedrusdata.catalog.spi.exception.CatalogInternalServerErrorException;
import ru.cedrusdata.catalog.spi.exception.CatalogPrincipalAlreadyExistsException;
import ru.cedrusdata.catalog.spi.exception.CatalogPrincipalDoesNotExistException;
import ru.cedrusdata.catalog.spi.model.AccessTokenCreateRequest;
import ru.cedrusdata.catalog.spi.model.AccessTokenCreateResponse;
import ru.cedrusdata.catalog.spi.model.AccessTokenCreateTemporaryResponse;
import ru.cedrusdata.catalog.spi.model.AccessTokenListResponse;
import ru.cedrusdata.catalog.spi.model.GrantRoleRequest;
import ru.cedrusdata.catalog.spi.model.PrincipalCreateRequest;
import ru.cedrusdata.catalog.spi.model.PrincipalInfo;
import ru.cedrusdata.catalog.spi.model.PrincipalListResponse;
import ru.cedrusdata.catalog.spi.model.PrincipalUpdateRequest;
import ru.cedrusdata.catalog.spi.model.RevokeRoleRequest;
import ru.cedrusdata.catalog.spi.model.RoleMemberInfo;
import ru.cedrusdata.catalog.spi.model.RoleMembersResponse;
import ru.cedrusdata.catalog.spi.model.UpdateResponse;
import ru.cedrusdata.catalog.store.PrincipalStore;

public class PrincipalService {
    private static final Logger logger = Logger.get(PrincipalService.class);
    private static final FileAttribute<?> INITIAL_ACCESS_TOKEN_FILE_ATTRIBUTES = PosixFilePermissions.asFileAttribute(Set.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE));
    private static final String BUILTIN_PREFIX = "builtin";
    public static final String CLIENT_ID_PREFIX = "U-";
    private final Authorizer authorizer;
    private final PrincipalStore store;
    private final PrincipalRoleIdService roleIdService;
    private final CatalogConfig config;
    private final JwtTokenManager jwtTokenManager;
    private final int jwtTtlSeconds;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final PrincipalServiceCache principalCache;
    private final PrivilegeInternalService privilegeInternalService;
    private final SecurityProviderService securityProviderService;

    @Inject
    public PrincipalService(CatalogConfig config, Authorizer authorizer, PrincipalStore store, PrincipalRoleIdService roleIdService, JwtTokenManager jwtTokenManager, PrincipalServiceCache principalCache, PrivilegeInternalService privilegeInternalService, SecurityProviderService securityProviderService) {
        this.authorizer = Objects.requireNonNull(authorizer, "authorizer");
        this.store = Objects.requireNonNull(store, "store");
        this.roleIdService = Objects.requireNonNull(roleIdService, "roleIdService");
        this.config = Objects.requireNonNull(config, "config");
        this.principalCache = Objects.requireNonNull(principalCache, "principalCache");
        this.jwtTokenManager = Objects.requireNonNull(jwtTokenManager, "jwtTokenManager");
        this.jwtTtlSeconds = Math.max((int)config.getJwtTokenTtl().roundTo(TimeUnit.SECONDS), 1);
        this.privilegeInternalService = Objects.requireNonNull(privilegeInternalService, "privilegeInternalService");
        this.securityProviderService = Objects.requireNonNull(securityProviderService, "securityPluginService");
    }

    private static String createAccessTokenId() {
        return CLIENT_ID_PREFIX + UUID.randomUUID().toString().replace("-", "");
    }

    private static String createAccessToken(String accessTokenId) {
        return accessTokenId + ":" + CatalogUtils.generateRandomString(72 - accessTokenId.length() - 1);
    }

    @PostConstruct
    public void initialize() {
        String hashedInitialAdminPassword;
        boolean printAccessToken;
        String initialAdminPrincipal;
        if (!this.initialized.compareAndSet(false, true)) {
            return;
        }
        if (this.store.countUserPrincipals() > 0L) {
            return;
        }
        try {
            initialAdminPrincipal = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(this.config.getAdminPrincipal());
        }
        catch (Exception e) {
            throw new CatalogInternalServerErrorException(String.format("Invalid property %s: %s", "admin-principal", e.getMessage()), (Throwable)e);
        }
        if (this.config.getAdminPassword() == null || this.config.getAdminPassword().isEmpty()) {
            printAccessToken = true;
            hashedInitialAdminPassword = CatalogUtils.hashPassword(CatalogUtils.generateRandomString(72));
        } else {
            printAccessToken = false;
            hashedInitialAdminPassword = CatalogUtils.hashPassword(this.config.getAdminPassword());
        }
        Optional<UUID> principalId = this.store.createPrincipalIfNotExists(initialAdminPrincipal, PrincipalType.USER, hashedInitialAdminPassword, Map.of(this.roleIdService.adminRoleId(), "builtin.admin"), true, SecurityProviderService.INTERNAL_PROVIDER_ID, Map.of(), this.roleIdService.adminRoleId());
        if (principalId.isPresent()) {
            logger.info("Created initial admin principal: " + initialAdminPrincipal);
            if (printAccessToken) {
                String accessToken = this.createAccessToken(this.principal(initialAdminPrincipal), new AccessTokenCreateRequest(initialAdminPrincipal, "Initial access token")).getAccessToken();
                this.printInitialAccessToken(accessToken);
            } else {
                logger.info("Skipping initial access token creation because an explicit admin password has been provided");
            }
        }
    }

    private void printInitialAccessToken(String accessToken) {
        Path path = this.config.getAdminAccessTokenFilePath().toPath();
        try {
            Files.deleteIfExists(path);
            Path parentPath = path.getParent();
            if (parentPath != null) {
                Files.createDirectories(parentPath, new FileAttribute[0]);
            }
            Files.createFile(path, INITIAL_ACCESS_TOKEN_FILE_ATTRIBUTES);
            Files.writeString(path, (CharSequence)accessToken, StandardOpenOption.TRUNCATE_EXISTING);
            logger.info("Created initial access token file: " + String.valueOf(path));
        }
        catch (IOException e) {
            logger.warn((Throwable)e, "Failed to create initial access token file, token will be printed to the log: " + e.getMessage());
            logger.info("Created initial access token: " + accessToken);
        }
    }

    @VisibleForTesting
    public String getAdminPrincipal() {
        return this.config.getAdminPrincipal();
    }

    @VisibleForTesting
    public String getAdminPassword() {
        return this.config.getAdminPassword();
    }

    public void createPrincipal(AuthenticatedPrincipal currentPrincipal, PrincipalCreateRequest request) {
        UUID identifiedWith;
        boolean active;
        String hashedPassword;
        this.authorizer.authorizePrincipalCreate(currentPrincipal);
        String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(request.getPrincipalName());
        if (PrincipalService.isBuiltinRole(normalizedPrincipalName)) {
            throw new CatalogBadRequestException(String.format("Principal name cannot start with \"%s\"", BUILTIN_PREFIX));
        }
        String principalType = request.getPrincipalType();
        PrincipalType normalizedPrincipalType = PrincipalType.resolveFromString(principalType).orElseThrow(() -> new CatalogBadRequestException("Unsupported principal type: \"" + principalType + "\""));
        if (normalizedPrincipalType == PrincipalType.USER) {
            String password = request.getPassword() != null ? CatalogUtils.normalizeNotEmpty(request.getPassword(), "Password cannot be empty") : null;
            hashedPassword = password != null ? CatalogUtils.hashPassword(password) : null;
            active = request.getActive() != null ? request.getActive() : true;
            identifiedWith = this.securityProviderService.resolveProviderId(currentPrincipal, Optional.ofNullable(request.getIdentifiedWith()));
        } else {
            if (request.getPassword() != null) {
                throw new CatalogBadRequestException("Role cannot have a password");
            }
            if (request.getActive() != null && !request.getActive().booleanValue()) {
                throw new CatalogBadRequestException("Role cannot be inactive");
            }
            if (request.getIdentifiedWith() != null) {
                throw new CatalogBadRequestException("Cannot change authentication for role");
            }
            hashedPassword = null;
            active = true;
            identifiedWith = null;
        }
        this.principalCache.withWriteLock(() -> {
            Optional<UUID> principalId = this.store.createPrincipalIfNotExists(normalizedPrincipalName, normalizedPrincipalType, hashedPassword, Map.of(), active, identifiedWith, request.getProperties(), currentPrincipal.id());
            if (principalId.isEmpty()) {
                throw new CatalogPrincipalAlreadyExistsException(normalizedPrincipalName);
            }
        });
    }

    public void updatePrincipal(AuthenticatedPrincipal currentPrincipal, String principalName, PrincipalUpdateRequest request) {
        Optional newIdentifiedWith;
        String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(principalName);
        if (PrincipalService.isBuiltinRole(normalizedPrincipalName)) {
            throw new CatalogBadRequestException("Cannot update built-in principal");
        }
        Optional newPrincipalName = request.getPrincipalName() != null ? Optional.of(CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(request.getPrincipalName().getValue())) : Optional.empty();
        Optional newPassword = request.getPassword() != null ? Optional.of(CatalogUtils.normalizeNotEmpty(request.getPassword().getValue(), "Password cannot be empty")) : Optional.empty();
        Optional newActive = request.getActive() != null ? Optional.of(request.getActive().getValue()) : Optional.empty();
        Map updatedProperties = request.getUpdatedProperties();
        Set removedProperties = request.getRemovedProperties();
        Optional<Object> optional = request.getIdentifiedWith() != null ? Optional.of(request.getIdentifiedWith().getValue() == null || request.getIdentifiedWith().getValue().isEmpty() ? "" : CatalogObjectNameValidation.VALIDATION_SECURITY_PROVIDER.normalizeObjectName(request.getIdentifiedWith().getValue())) : (newIdentifiedWith = Optional.empty());
        if (newPrincipalName.isEmpty() && newPassword.isEmpty() && newActive.isEmpty() && (updatedProperties == null || updatedProperties.isEmpty()) && (removedProperties == null || removedProperties.isEmpty()) && newIdentifiedWith.isEmpty()) {
            throw new CatalogBadRequestException("Nothing to update");
        }
        if (newPrincipalName.isPresent() && PrincipalService.isBuiltinRole((String)newPrincipalName.get())) {
            throw new CatalogBadRequestException(String.format("Principal name cannot start with \"%s\"", BUILTIN_PREFIX));
        }
        this.principalCache.withWriteLock(() -> {
            Optional<PrincipalDetails> principal = this.store.getPrincipalDetailsByName(normalizedPrincipalName, false);
            if (principal.isEmpty()) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            this.authorizer.authorizePrincipalAlter(currentPrincipal, principal.get().toSecurable());
            if (newPassword.isPresent() && principal.get().type() == PrincipalType.ROLE) {
                throw new CatalogBadRequestException("Cannot change password for role");
            }
            if (newActive.isPresent() && principal.get().type() == PrincipalType.ROLE) {
                throw new CatalogBadRequestException("Cannot change active flag for role");
            }
            UUID resolvedIdentifiedWith = principal.get().identifiedWith().orElse(null);
            if (newIdentifiedWith.isPresent()) {
                if (principal.get().type() == PrincipalType.ROLE) {
                    throw new CatalogBadRequestException("Cannot change authentication for role");
                }
                resolvedIdentifiedWith = this.securityProviderService.resolveProviderId(currentPrincipal, newIdentifiedWith);
            }
            String resolvedPrincipalName = newPrincipalName.orElse(principal.get().name());
            boolean resolvedActive = newActive.orElse(principal.get().active());
            Map<String, String> resolvedProperties = CatalogUtils.mergeProperties(principal.get().properties(), updatedProperties, removedProperties);
            boolean updated = this.store.updatePrincipalIfExists(principal.get().id(), resolvedPrincipalName, resolvedActive, resolvedIdentifiedWith, resolvedProperties, newPassword.map(v -> CatalogUtils.hashPassword(v.trim())));
            if (!updated) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            List<String> accessTokenIds = this.store.listAccessTokenIds(principal.get().id());
            this.principalCache.invalidatePrincipalNameAndAccessTokenIds(normalizedPrincipalName, accessTokenIds);
        });
    }

    public void deletePrincipal(AuthenticatedPrincipal currentPrincipal, String principalName) {
        String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(principalName);
        if (PrincipalService.isBuiltinRole(normalizedPrincipalName)) {
            throw new CatalogBadRequestException("Cannot delete built-in principal");
        }
        Optional<PrincipalDetails> principalDetails = this.store.getPrincipalDetailsByName(normalizedPrincipalName, false);
        if (principalDetails.isEmpty()) {
            throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
        }
        UUID principalId = principalDetails.get().id();
        if (currentPrincipal.id().equals(principalId)) {
            throw new CatalogBadRequestException("Cannot delete self");
        }
        this.authorizer.authorizePrincipalDrop(currentPrincipal, principalDetails.get().toSecurable());
        this.principalCache.withWriteLock(() -> {
            List<String> accessTokenIds = this.store.listAccessTokenIds(principalId);
            boolean deleted = this.store.deletePrincipalIfExists(principalId, normalizedPrincipalName);
            if (!deleted) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            this.principalCache.invalidatePrincipalNameAndAccessTokenIds(normalizedPrincipalName, accessTokenIds);
        });
        this.privilegeInternalService.clearPrivileges();
    }

    public PrincipalInfo getPrincipal(AuthenticatedPrincipal currentPrincipal, String principalName) {
        String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(principalName);
        Optional<PrincipalInfoEx> info = this.store.getPrincipal(normalizedPrincipalName);
        if (info.isEmpty()) {
            throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
        }
        this.authorizer.authorizePrincipalDescribe(currentPrincipal, info.get().toSecurable());
        return info.get().info();
    }

    public PrincipalListResponse listPrincipals(AuthenticatedPrincipal currentPrincipal, Optional<String> principalType, Optional<Boolean> active, ResultPage page) {
        AuthorizerPrincipalPredicate predicate = this.authorizer.authorizePrincipalList(currentPrincipal);
        PageData<PrincipalInfoEx> pageData = this.store.listPrincipals(principalType, active, page, (Predicate<PrincipalInfoEx>)((Predicate)info -> predicate.test(info.toSecurable())));
        return new PrincipalListResponse(pageData.pageData().stream().map(PrincipalInfoEx::info).toList(), pageData.nextPageToken());
    }

    public boolean grantOwnership(AuthenticatedPrincipal currentPrincipal, String principalName, AuthenticatedPrincipal newOwnerPrincipal) {
        return this.principalCache.withWriteLock(() -> {
            String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(principalName);
            Optional<PrincipalDetails> principal = this.store.getPrincipalDetailsByName(normalizedPrincipalName, false);
            if (principal.isEmpty()) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            this.authorizer.authorizeGrantOwnership(currentPrincipal, principal.get().toSecurable());
            if (principal.get().ownerId().equals(Optional.of(newOwnerPrincipal.id()))) {
                return false;
            }
            boolean updated = this.store.updatePrincipalOwnerIfExists(principal.get().id(), newOwnerPrincipal.id());
            if (!updated) {
                throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
            }
            List<String> accessTokenIds = this.store.listAccessTokenIds(principal.get().id());
            this.principalCache.invalidatePrincipalNameAndAccessTokenIds(normalizedPrincipalName, accessTokenIds);
            return true;
        });
    }

    public UpdateResponse grantRole(AuthenticatedPrincipal currentPrincipal, GrantRoleRequest request) {
        if (request.getRoles() == null || request.getRoles().isEmpty()) {
            throw new CatalogBadRequestException("No roles provided");
        }
        String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(request.getPrincipalName());
        Set normalizedRoleNames = request.getRoles().stream().map(CatalogObjectNameValidation.VALIDATION_ROLE::normalizeObjectName).collect(Collectors.toSet());
        if ("builtin.public".equals(normalizedPrincipalName)) {
            throw new CatalogBadRequestException(String.format("Cannot grant role to \"%s\"", "builtin.public"));
        }
        return this.principalCache.withWriteLock(() -> {
            PrincipalDetails principal = this.store.getPrincipalDetailsByName(normalizedPrincipalName, false).orElseThrow(() -> new CatalogPrincipalDoesNotExistException(normalizedPrincipalName));
            HashMap<UUID, String> roles = new HashMap<UUID, String>();
            for (String roleName : normalizedRoleNames) {
                PrincipalDetails role = this.store.getPrincipalDetailsByName(roleName, false).orElseThrow(() -> new CatalogPrincipalDoesNotExistException(roleName));
                this.authorizer.authorizeRoleGrant(currentPrincipal, role.toSecurable());
                if (role.type() == PrincipalType.USER) {
                    throw new CatalogBadRequestException(String.format("\"%s\" references a user", roleName));
                }
                if (role.id().equals(principal.id())) {
                    throw new CatalogBadRequestException("Role and principal cannot be the same");
                }
                if (role.id().equals(this.roleIdService.publicRoleId())) continue;
                roles.put(role.id(), role.name());
            }
            boolean updated = this.store.grantRoles(principal.id(), principal.name(), roles);
            if (!updated) {
                return new UpdateResponse(Boolean.valueOf(false));
            }
            List<String> accessTokenIds = this.store.listAccessTokenIds(principal.id());
            this.principalCache.invalidatePrincipalNameAndAccessTokenIds(normalizedPrincipalName, accessTokenIds);
            return new UpdateResponse(Boolean.valueOf(true));
        });
    }

    public UpdateResponse revokeRole(AuthenticatedPrincipal currentPrincipal, RevokeRoleRequest request) {
        if (request.getRoles() == null || request.getRoles().isEmpty()) {
            throw new CatalogBadRequestException("No roles provided");
        }
        String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(request.getPrincipalName());
        Set normalizedRoleNames = request.getRoles().stream().map(CatalogObjectNameValidation.VALIDATION_ROLE::normalizeObjectName).collect(Collectors.toSet());
        if ("builtin.public".equals(normalizedPrincipalName)) {
            throw new CatalogBadRequestException(String.format("Cannot revoke role from \"%s\"", "builtin.public"));
        }
        return this.principalCache.withWriteLock(() -> {
            PrincipalDetails principal = this.store.getPrincipalDetailsByName(normalizedPrincipalName, false).orElseThrow(() -> new CatalogPrincipalDoesNotExistException(normalizedPrincipalName));
            HashSet<UUID> roleIds = new HashSet<UUID>();
            for (String roleName : normalizedRoleNames) {
                PrincipalDetails role = this.store.getPrincipalDetailsByName(roleName, false).orElseThrow(() -> new CatalogPrincipalDoesNotExistException(roleName));
                this.authorizer.authorizeRoleRevoke(currentPrincipal, role.toSecurable());
                if (role.type() == PrincipalType.USER) {
                    throw new CatalogBadRequestException(String.format("\"%s\" references a user", roleName));
                }
                if (role.id().equals(principal.id())) {
                    throw new CatalogBadRequestException("Role and principal cannot be the same");
                }
                if (role.id().equals(this.roleIdService.publicRoleId())) {
                    throw new CatalogBadRequestException(String.format("Cannot revoke %s role", "builtin.public"));
                }
                UUID adminRoleId = this.roleIdService.adminRoleId();
                if (role.id().equals(adminRoleId) && currentPrincipal.id().equals(principal.id()) && currentPrincipal.roleIds().contains(adminRoleId)) {
                    throw new CatalogBadRequestException(String.format("Cannot revoke role \"%s\" from self", "builtin.admin"));
                }
                roleIds.add(role.id());
            }
            boolean updated = this.store.revokeRoles(principal.id(), roleIds);
            if (!updated) {
                return new UpdateResponse(Boolean.valueOf(false));
            }
            List<String> accessTokenIds = this.store.listAccessTokenIds(principal.id());
            this.principalCache.invalidatePrincipalNameAndAccessTokenIds(normalizedPrincipalName, accessTokenIds);
            return new UpdateResponse(Boolean.valueOf(true));
        });
    }

    public RoleMembersResponse roleMembers(AuthenticatedPrincipal currentPrincipal, String roleName) {
        String normalizedRoleName = CatalogObjectNameValidation.VALIDATION_ROLE.normalizeObjectName(roleName);
        PrincipalDetails role = this.store.getPrincipalDetailsByName(normalizedRoleName, false).orElseThrow(() -> new CatalogPrincipalDoesNotExistException(normalizedRoleName));
        this.authorizer.authorizeRoleListMembers(currentPrincipal, role.toSecurable());
        if (role.type() != PrincipalType.ROLE) {
            throw new CatalogBadRequestException(String.format("Principal \"%s\" is not a role", roleName));
        }
        List<RoleMemberInfo> items = this.store.roleMembers(role.id(), this.roleIdService.publicRoleId().equals(role.id()));
        return new RoleMembersResponse(items);
    }

    public AccessTokenCreateTemporaryResponse createTemporaryAccessToken(AuthenticationContext actorContext, String subject) {
        AuthenticationContext impersonatedContext = this.impersonate(actorContext, subject);
        String subjectToken = this.jwtTokenManager.wrapCredentials(new JwtCredentials(impersonatedContext.actor().name(), impersonatedContext.subject().name()), this.jwtTtlSeconds);
        return new AccessTokenCreateTemporaryResponse(subject, subjectToken, Integer.valueOf(this.jwtTtlSeconds));
    }

    public Optional<JwtCredentials> getTemporaryAccessTokenCredentials(String token) {
        return this.jwtTokenManager.unwrapCredentials(token);
    }

    public AccessTokenCreateResponse createAccessToken(AuthenticatedPrincipal currentPrincipal, AccessTokenCreateRequest request) {
        String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(request.getPrincipalName());
        Optional<PrincipalDetails> principal = this.store.getPrincipalDetailsByName(normalizedPrincipalName, false);
        if (principal.isEmpty()) {
            throw new CatalogPrincipalDoesNotExistException(normalizedPrincipalName);
        }
        this.authorizer.authorizePrincipalCreateAccessToken(currentPrincipal, principal.get().toSecurable());
        if (principal.get().type() != PrincipalType.USER) {
            throw new CatalogBadRequestException("Cannot create access token for " + principal.get().type().getCaption());
        }
        String accessTokenId = PrincipalService.createAccessTokenId();
        String accessToken = PrincipalService.createAccessToken(accessTokenId);
        String hashedAccessToken = CatalogUtils.hashPassword(accessToken);
        this.principalCache.withWriteLock(() -> this.store.createAccessToken(((PrincipalDetails)principal.get()).id(), normalizedPrincipalName, accessTokenId, hashedAccessToken, request.getDescription()));
        return new AccessTokenCreateResponse(accessTokenId, accessToken);
    }

    public void deleteAccessToken(AuthenticatedPrincipal currentPrincipal, String accessTokenId) {
        String normalizedAccessTokenId = CatalogUtils.normalizeNotEmpty(accessTokenId, "Access token ID cannot be empty");
        Optional<PrincipalDetails> principal = this.store.getPrincipalDetailsByAccessTokenId(normalizedAccessTokenId);
        if (principal.isEmpty()) {
            throw new CatalogAccessTokenDoesNotExistException(normalizedAccessTokenId);
        }
        this.authorizer.authorizePrincipalDropAccessToken(currentPrincipal, principal.get().toSecurable());
        this.principalCache.withWriteLock(() -> {
            boolean deleted = this.store.deleteAccessTokenIfExists(normalizedAccessTokenId);
            if (!deleted) {
                throw new CatalogAccessTokenDoesNotExistException(normalizedAccessTokenId);
            }
            this.principalCache.invalidateAccessTokenId(normalizedAccessTokenId);
        });
    }

    public AccessTokenListResponse listAccessTokens(AuthenticatedPrincipal currentPrincipal, Optional<String> principalName, ResultPage page) {
        String normalizePrincipalName;
        AuthorizerPrincipalPredicate authorizerPredicate = this.authorizer.authorizePrincipalListAccessTokens(currentPrincipal);
        Predicate predicate = principal -> authorizerPredicate.test(principal.toSecurable());
        Optional<UUID> principalId = Optional.empty();
        if (principalName.isPresent() && (principalId = this.store.getPrincipalIdByName(normalizePrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(principalName.get()))).isEmpty()) {
            return new AccessTokenListResponse(List.of(), null);
        }
        return this.store.listAccessTokens(page, principalId, (Predicate<AccessTokenInfoEx>)predicate);
    }

    public Optional<AuthenticatedPrincipal> authenticate(String principalName) {
        return this.authenticate(principalName, Optional.empty());
    }

    public Optional<AuthenticatedPrincipal> authenticate(UUID principalId) {
        Optional<PrincipalDetails> principalDetails = this.store.getPrincipalDetailsById(principalId);
        return this.authenticate(principalDetails, Optional.empty());
    }

    public Optional<AuthenticatedPrincipal> authenticate(String principalName, String password) {
        password = CatalogUtils.normalizeNotEmpty(password, "Password cannot be empty");
        return this.authenticate(principalName, Optional.of(password));
    }

    private Optional<AuthenticatedPrincipal> authenticate(String principalName, Optional<String> password) {
        return this.principalCache.withReadLock(() -> {
            Optional<PrincipalServiceCacheEntry> cached;
            String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(principalName);
            if (password.isEmpty() && (cached = this.principalCache.getByPrincipalName(normalizedPrincipalName)).isPresent()) {
                if (!this.cachedPrincipalIsValid(cached.get())) {
                    return Optional.empty();
                }
                return cached.map(PrincipalServiceCacheEntry::toPrincipal);
            }
            Optional<PrincipalDetails> principalDetails = this.store.getPrincipalDetailsByName(normalizedPrincipalName, true);
            Optional<AuthenticatedPrincipal> principal = this.authenticate(principalDetails, password);
            if (principal.isPresent()) {
                this.principalCache.putPrincipalName(principal.get().toCacheEntry(null, principalDetails.get().identifiedWith().get()));
            }
            return principal;
        });
    }

    private Optional<AuthenticatedPrincipal> authenticate(Optional<PrincipalDetails> principalDetails, Optional<String> password) {
        if (principalDetails.isEmpty()) {
            return Optional.empty();
        }
        if (!principalDetails.get().active()) {
            return Optional.empty();
        }
        if (principalDetails.get().type() == PrincipalType.ROLE) {
            return Optional.empty();
        }
        if (!this.checkUserPassword(principalDetails.get(), password)) {
            return Optional.empty();
        }
        return principalDetails.map(details -> details.toPrincipal(this.roleIdService.publicRoleId()));
    }

    public Optional<AuthenticatedPrincipal> authenticateAccessToken(String accessToken) {
        if (accessToken == null) {
            return Optional.empty();
        }
        List parts = Splitter.on((char)':').splitToList((CharSequence)accessToken);
        if (parts.size() != 2) {
            return Optional.empty();
        }
        String accessTokenId = ((String)parts.get(0)).trim();
        if (accessTokenId.isEmpty()) {
            return Optional.empty();
        }
        return this.principalCache.withReadLock(() -> {
            Optional<PrincipalServiceCacheEntry> cached = this.principalCache.getByAccessTokenId(accessTokenId, accessToken);
            if (cached.isPresent()) {
                if (!this.cachedPrincipalIsValid(cached.get())) {
                    return Optional.empty();
                }
                return cached.map(PrincipalServiceCacheEntry::toPrincipal);
            }
            Optional<PrincipalDetails> principalDetails = this.store.getPrincipalDetailsByAccessTokenId(accessTokenId);
            if (principalDetails.isEmpty()) {
                return Optional.empty();
            }
            if (!principalDetails.get().active()) {
                return Optional.empty();
            }
            if (!CatalogUtils.passwordMatches(accessToken, principalDetails.get().hashedPassword())) {
                return Optional.empty();
            }
            if (!this.checkUserPassword(principalDetails.get(), Optional.empty())) {
                return Optional.empty();
            }
            AuthenticatedPrincipal principal = principalDetails.get().toPrincipal(this.roleIdService.publicRoleId());
            this.principalCache.putAccessTokenId(accessTokenId, principal.toCacheEntry(accessToken, principalDetails.get().identifiedWith().get()));
            return Optional.of(principal);
        });
    }

    private boolean cachedPrincipalIsValid(PrincipalServiceCacheEntry cached) {
        UUID securityProviderId = cached.identifiedWith();
        if (!this.securityProviderService.useCustomAuthentication(securityProviderId)) {
            return true;
        }
        return this.securityProviderService.authenticate(securityProviderId, cached.name(), Optional.empty());
    }

    private boolean checkUserPassword(PrincipalDetails principalDetails, Optional<String> password) {
        Verify.verify((boolean)principalDetails.identifiedWith().isPresent());
        UUID securityProviderId = principalDetails.identifiedWith().get();
        if (this.securityProviderService.useCustomAuthentication(securityProviderId)) {
            return this.securityProviderService.authenticate(securityProviderId, principalDetails.name(), password);
        }
        return password.isEmpty() || CatalogUtils.passwordMatches(password.get(), principalDetails.hashedPassword());
    }

    public AuthenticationContext impersonate(AuthenticationContext context, String subject) {
        AuthenticatedPrincipal subjectPrincipal = this.authenticate(subject).orElseThrow(CatalogAuthenticationException::new);
        return this.impersonate(context, subjectPrincipal);
    }

    public AuthenticationContext impersonate(AuthenticationContext context, UUID subjectId) {
        AuthenticatedPrincipal subjectPrincipal = this.authenticate(subjectId).orElseThrow(CatalogAuthenticationException::new);
        return this.impersonate(context, subjectPrincipal);
    }

    public AuthenticationContext impersonate(AuthenticationContext context, AuthenticatedPrincipal subjectPrincipal) {
        if (context.subject().id().equals(subjectPrincipal.id())) {
            return context;
        }
        if (context.isImpersonated()) {
            throw new CatalogAuthenticationException();
        }
        AuthenticatedPrincipal actorPrincipal = context.actor();
        this.authorizer.authorizePrincipalImpersonate(actorPrincipal, subjectPrincipal.toSecurable());
        return new AuthenticationContext(actorPrincipal, subjectPrincipal);
    }

    public AuthenticatedPrincipal principal(String principalName) {
        String normalizedPrincipalName = CatalogObjectNameValidation.VALIDATION_PRINCIPAL.normalizeObjectName(principalName);
        PrincipalDetails principal = this.store.getPrincipalDetailsByName(normalizedPrincipalName, false).orElseThrow(() -> new CatalogPrincipalDoesNotExistException(normalizedPrincipalName));
        return principal.toPrincipal(this.roleIdService.publicRoleId());
    }

    public static boolean isBuiltinRole(String normalizedPrincipalName) {
        return normalizedPrincipalName.startsWith(BUILTIN_PREFIX);
    }
}

