/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.core.maintenance;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.Striped;
import com.google.inject.Inject;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import jakarta.annotation.PostConstruct;
import java.time.DateTimeException;
import java.time.ZoneId;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import ru.cedrusdata.catalog.CatalogObjectNameValidation;
import ru.cedrusdata.catalog.CatalogUtils;
import ru.cedrusdata.catalog.CurrentTimeSupplier;
import ru.cedrusdata.catalog.config.CatalogMaintenanceConfig;
import ru.cedrusdata.catalog.core.PageData;
import ru.cedrusdata.catalog.core.ResultPageIterator;
import ru.cedrusdata.catalog.core.UsageAwareResourceSession;
import ru.cedrusdata.catalog.core.computeengine.ComputeEngineService;
import ru.cedrusdata.catalog.core.computeengine.ComputeEngineSession;
import ru.cedrusdata.catalog.core.maintenance.MaintenanceJobInfoEx;
import ru.cedrusdata.catalog.core.maintenance.MaintenanceOperation;
import ru.cedrusdata.catalog.core.maintenance.MaintenanceOperationExecutor;
import ru.cedrusdata.catalog.core.maintenance.MaintenanceOperationInfoEx;
import ru.cedrusdata.catalog.core.maintenance.MaintenanceOperationObject;
import ru.cedrusdata.catalog.core.maintenance.MaintenanceOperationObjectGroup;
import ru.cedrusdata.catalog.core.maintenance.MaintenanceOperationResult;
import ru.cedrusdata.catalog.core.maintenance.MaintenanceOperationTarget;
import ru.cedrusdata.catalog.core.maintenance.ScheduleExecutionTime;
import ru.cedrusdata.catalog.core.maintenance.ScheduleTimeResolver;
import ru.cedrusdata.catalog.core.objectgroup.ObjectGroupService;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.core.principal.AuthenticationContext;
import ru.cedrusdata.catalog.core.principal.PrincipalService;
import ru.cedrusdata.catalog.core.security.authorization.Authorizer;
import ru.cedrusdata.catalog.core.security.authorization.PrivilegeInternalService;
import ru.cedrusdata.catalog.core.security.authorization.predicate.AuthorizerComputeEnginePredicate;
import ru.cedrusdata.catalog.core.security.authorization.predicate.AuthorizerJobPredicate;
import ru.cedrusdata.catalog.iceberg.table.IcebergTableService;
import ru.cedrusdata.catalog.spi.client.ResultPage;
import ru.cedrusdata.catalog.spi.computeengine.CatalogComputeEngine;
import ru.cedrusdata.catalog.spi.computeengine.CatalogComputeEngineOperationFactory;
import ru.cedrusdata.catalog.spi.exception.CatalogAuthenticationException;
import ru.cedrusdata.catalog.spi.exception.CatalogBadRequestException;
import ru.cedrusdata.catalog.spi.exception.CatalogInternalServerErrorException;
import ru.cedrusdata.catalog.spi.exception.CatalogMaintenanceJobAlreadyExistsException;
import ru.cedrusdata.catalog.spi.exception.CatalogMaintenanceJobDoesNotExistException;
import ru.cedrusdata.catalog.spi.exception.CatalogMaintenanceOperationDoesNotExistException;
import ru.cedrusdata.catalog.spi.model.JobScheduleConfig;
import ru.cedrusdata.catalog.spi.model.MaintenanceJobCreateRequest;
import ru.cedrusdata.catalog.spi.model.MaintenanceJobInfo;
import ru.cedrusdata.catalog.spi.model.MaintenanceJobListResponse;
import ru.cedrusdata.catalog.spi.model.MaintenanceJobUpdateRequest;
import ru.cedrusdata.catalog.spi.model.MaintenanceOperationConfig;
import ru.cedrusdata.catalog.spi.model.MaintenanceOperationInfo;
import ru.cedrusdata.catalog.spi.model.MaintenanceOperationListResponse;
import ru.cedrusdata.catalog.spi.model.MaintenanceOperationStartRequest;
import ru.cedrusdata.catalog.spi.model.MaintenanceOperationStartResponse;
import ru.cedrusdata.catalog.store.MaintenanceJobStore;
import ru.cedrusdata.catalog.store.MaintenanceOperationStore;
import ru.cedrusdata.catalog.store.jdbc.QueryCondition;

public class MaintenanceOperationService {
    private static final String CANCEL_ON_SHUTDOWN_MESSAGE = new CancellationException("Cancelled due to server shutdown").toString();
    private static final ZoneId UTC_TIME_ZONE = ZoneId.of("UTC");
    private static final Logger log = Logger.get(MaintenanceOperationService.class);
    private final CatalogMaintenanceConfig config;
    private final Authorizer authorizer;
    private final ObjectGroupService objectGroupService;
    private final IcebergTableService tableService;
    private final ComputeEngineService computeEngineService;
    private final PrincipalService principalService;
    private final MaintenanceOperationStore operationStore;
    private final MaintenanceJobStore jobStore;
    private final CurrentTimeSupplier currentTimeSupplier;
    private final AtomicInteger activeOperationsCounter = new AtomicInteger();
    private final ConcurrentHashMap<UUID, MaintenanceOperation> activeOperationsById = new ConcurrentHashMap();
    private final Striped<Lock> jobLock = Striped.lock((int)64);
    private final ConcurrentHashMap<UUID, ActiveJob> scheduledJobsById = new ConcurrentHashMap();
    private final ScheduledExecutorService cleanupExecutor = Executors.newSingleThreadScheduledExecutor(Threads.daemonThreadsNamed((String)"maintenance-history-cleanup"));
    private final ScheduledThreadPoolExecutor scheduledExecutor = new ScheduledThreadPoolExecutor(1, Threads.daemonThreadsNamed((String)"maintenance-schedule-worker"));
    private final MaintenanceOperationExecutor operationExecutor;
    private final BiFunction<JobScheduleConfig, Long, Long> nextExecutionTime = (scheduleConfig, lastExecutionTime) -> {
        ScheduleTimeResolver scheduleTimeResolver;
        try {
            scheduleTimeResolver = this.scheduleTimeResolver((JobScheduleConfig)scheduleConfig);
        }
        catch (CatalogBadRequestException e) {
            log.error((Throwable)e, "Failed to parse stored job schedule");
            return null;
        }
        return this.nextExecutionTime(scheduleTimeResolver, lastExecutionTime != null ? new ScheduleExecutionTime((long)lastExecutionTime, 0L) : null).timestamp();
    };
    private final AtomicBoolean initGuard = new AtomicBoolean();
    private final PrivilegeInternalService privilegeInternalService;

    @Inject
    public MaintenanceOperationService(CatalogMaintenanceConfig config, Authorizer authorizer, ObjectGroupService objectGroupService, IcebergTableService tableService, ComputeEngineService computeEngineService, PrincipalService principalService, MaintenanceOperationStore operationStore, MaintenanceJobStore jobStore, CurrentTimeSupplier currentTimeSupplier, PrivilegeInternalService privilegeInternalService) {
        this.config = config;
        this.authorizer = authorizer;
        this.objectGroupService = objectGroupService;
        this.tableService = tableService;
        this.computeEngineService = computeEngineService;
        this.principalService = principalService;
        this.operationStore = operationStore;
        this.jobStore = jobStore;
        this.currentTimeSupplier = currentTimeSupplier;
        this.operationExecutor = new MaintenanceOperationExecutor(config.getMaintenanceMaxActiveOperations(), Threads.daemonThreadsNamed((String)"maintenance-worker-%d"));
        this.scheduledExecutor.setRemoveOnCancelPolicy(true);
        this.privilegeInternalService = Objects.requireNonNull(privilegeInternalService, "privilegeInternalService");
    }

    @VisibleForTesting
    int scheduledQueueSize() {
        return this.scheduledExecutor.getQueue().size();
    }

    @PostConstruct
    public void init() {
        if (!this.initGuard.compareAndSet(false, true)) {
            return;
        }
        int count = this.operationStore.completeAllActive(this.currentTimeSupplier.currentTimeMillis(), CANCEL_ON_SHUTDOWN_MESSAGE);
        if (count > 0) {
            log.info("Marked %d maintenance operation(s) as CANCELLED due to server restart", new Object[]{count});
        }
        this.startStoredJobs();
        long historyMaxAge = this.config.getMaintenanceHistoryMaxAge().toMillis();
        long historyCleanupInterval = this.config.getMaintenanceHistoryCleanupInterval().toMillis();
        this.cleanupExecutor.scheduleWithFixedDelay(() -> this.cleanup(historyMaxAge), historyCleanupInterval, historyCleanupInterval, TimeUnit.MILLISECONDS);
    }

    private void startStoredJobs() {
        ResultPageIterator iterator = new ResultPageIterator(page -> this.jobStore.list((ResultPage)page, job -> true, List.of(), this.nextExecutionTime), 100);
        while (iterator.hasNext()) {
            ScheduleTimeResolver scheduleTimeResolver;
            MaintenanceJobInfoEx info = (MaintenanceJobInfoEx)iterator.next();
            try {
                scheduleTimeResolver = this.scheduleTimeResolver(info.info().getScheduleConfig());
            }
            catch (CatalogBadRequestException e) {
                log.error((Throwable)e, "Failed to parse stored job schedule");
                continue;
            }
            this.startJobScheduling(new ActiveJob(info.id(), scheduleTimeResolver));
        }
    }

    private void cleanup(long historyMaxAge) {
        long currentTime = this.currentTimeSupplier.currentTimeMillis();
        long minCompletedAt = currentTime - historyMaxAge;
        int count = this.operationStore.deleteCompleted(minCompletedAt);
        if (count > 0) {
            log.debug("Removed %d old history entries", new Object[]{count});
        }
    }

    public MaintenanceOperationStartResponse startOperation(AuthenticatedPrincipal principal, MaintenanceOperationStartRequest request) {
        UsageAwareResourceSession computeEngineSession = null;
        try {
            MaintenanceOperationConfig operationConfig = request.getOperationConfig();
            if (operationConfig == null) {
                throw new CatalogBadRequestException("Operation config cannot be empty");
            }
            String operationName = CatalogUtils.normalizeNotEmpty(operationConfig.getOperationName(), "Operation name cannot be empty");
            String engineName = operationConfig.getEngineName() != null ? CatalogObjectNameValidation.VALIDATION_COMPUTE_ENGINE.normalizeObjectName(operationConfig.getEngineName()) : "local";
            computeEngineSession = this.computeEngineService.createSession(engineName);
            this.authorizer.authorizeComputeEngineStartOperation(principal, ((ComputeEngineSession)computeEngineSession).toSecurable());
            CatalogComputeEngineOperationFactory operationFactory = ((CatalogComputeEngine)computeEngineSession.resource()).createOperationFactory(operationName, Optional.ofNullable(operationConfig.getParameters()).orElse(Map.of()));
            MaintenanceOperationTarget target = this.resolveTarget(principal, request);
            UUID operationId = this.startOperationInternal(principal, operationFactory, target, operationConfig, (ComputeEngineSession)computeEngineSession, null);
            return new MaintenanceOperationStartResponse(operationId);
        }
        catch (Throwable t) {
            if (computeEngineSession != null) {
                computeEngineSession.close();
            }
            throw t;
        }
    }

    private UUID startOperationInternal(AuthenticatedPrincipal principal, CatalogComputeEngineOperationFactory operationFactory, MaintenanceOperationTarget target, MaintenanceOperationConfig operationConfig, ComputeEngineSession computeEngineSession, ActiveJob job) {
        int activeOperations;
        UUID operationId = UUID.randomUUID();
        MaintenanceOperation operation = new MaintenanceOperation(this.config.getMaintenanceMaxOperationErrors(), computeEngineSession, principal, operationId, operationFactory, this.objectGroupService, this.tableService, target, operationConfig.getEngineConfig());
        do {
            if ((activeOperations = this.activeOperationsCounter.get()) < this.config.getMaintenanceMaxActiveOperations()) continue;
            throw new CatalogInternalServerErrorException(String.format("Too many concurrent maintenance operations (max %d)", this.config.getMaintenanceMaxActiveOperations()));
        } while (!this.activeOperationsCounter.compareAndSet(activeOperations, activeOperations + 1));
        this.activeOperationsById.put(operationId, operation);
        this.operationStore.create(computeEngineSession.engineId(), operationId, operationConfig.getOperationName(), operationConfig.getParameters(), Optional.ofNullable(job != null ? job.id : null), target.objectGroup().map(MaintenanceOperationObjectGroup::objectGroupId), target.object().map(MaintenanceOperationObject::objectId), operationConfig.getEngineConfig(), principal.id(), this.currentTimeSupplier.currentTimeMillis());
        ((CompletableFuture)operation.future()).whenComplete((result, error) -> this.onOperationCompleted(operationId, (MaintenanceOperationResult)result, (Throwable)error, job));
        this.operationExecutor.execute(operation);
        return operationId;
    }

    @VisibleForTesting
    public UUID startTestOperation(AuthenticatedPrincipal principal, String engineName, String operationName, Map<String, String> operationParameters) {
        UUID operationId = UUID.randomUUID();
        try (ComputeEngineSession session = this.computeEngineService.createSession(engineName);){
            CatalogComputeEngineOperationFactory operationFactory = ((CatalogComputeEngine)session.resource()).createOperationFactory(operationName, Optional.ofNullable(operationParameters).orElse(Map.of()));
            MaintenanceOperation operation = new MaintenanceOperation(this.config.getMaintenanceMaxOperationErrors(), session, principal, operationId, operationFactory, this.objectGroupService, this.tableService, new MaintenanceOperationTarget(Optional.empty(), Optional.of(new MaintenanceOperationObject("dummy", "dummy", "dummy", UUID.randomUUID()))), Map.of());
            this.activeOperationsById.put(operationId, operation);
            this.operationStore.create(session.engineId(), operationId, operationName, operationParameters, Optional.empty(), Optional.empty(), Optional.empty(), null, principal.id(), this.currentTimeSupplier.currentTimeMillis());
            ((CompletableFuture)operation.future()).whenComplete((result, error) -> this.onOperationCompleted(operationId, (MaintenanceOperationResult)result, (Throwable)error, null));
        }
        return operationId;
    }

    private void onOperationCompleted(UUID operationId, MaintenanceOperationResult result, Throwable error, ActiveJob job) {
        this.activeOperationsCounter.decrementAndGet();
        this.activeOperationsById.remove(operationId);
        if (error instanceof ExecutionException) {
            ExecutionException executionException = (ExecutionException)error;
            error = executionException.getCause();
        }
        if (error == null) {
            log.debug(String.format("Maintenance operation %s completed successfully", operationId));
        } else {
            log.warn(String.format("Maintenance operation %s failed: %s", operationId, error.getMessage()));
        }
        if (error != null) {
            this.operationStore.completeActive(operationId, this.currentTimeSupplier.currentTimeMillis(), Optional.empty(), Optional.empty(), Optional.of(error.getMessage()));
        } else {
            this.operationStore.completeActive(operationId, this.currentTimeSupplier.currentTimeMillis(), Optional.ofNullable(result.result()), Optional.ofNullable(result.objectErrors()), Optional.ofNullable(result.error()));
        }
        if (job != null) {
            this.withJobLock(job.id, currentJob -> {
                if (currentJob != job) {
                    return;
                }
                currentJob.scheduleNextExecution();
            });
        }
    }

    private MaintenanceOperationTarget resolveTarget(AuthenticatedPrincipal principal, MaintenanceOperationStartRequest request) {
        Optional<String> normalizedObjectGroupName = CatalogObjectNameValidation.VALIDATION_OBJECT_GROUP.normalizeOptionalObjectName(request.getTargetObjectGroupName());
        Optional<String> normalizedCatalogName = CatalogObjectNameValidation.VALIDATION_CATALOG.normalizeOptionalObjectName(request.getTargetCatalogName());
        Optional<String> normalizedNamespaceName = CatalogObjectNameValidation.VALIDATION_NAMESPACE.normalizeOptionalObjectName(request.getTargetNamespaceName());
        Optional<String> normalizedObjectName = CatalogObjectNameValidation.VALIDATION_OBJECT.normalizeOptionalObjectName(request.getTargetObjectName());
        if (normalizedObjectGroupName.isPresent()) {
            if (normalizedCatalogName.isPresent() || normalizedNamespaceName.isPresent() || normalizedObjectName.isPresent()) {
                throw new CatalogBadRequestException("Either object group name or object name must be provided");
            }
            UUID objectGroupId = this.objectGroupService.resolveIdForMaintenance(normalizedObjectGroupName.get());
            return new MaintenanceOperationTarget(Optional.of(new MaintenanceOperationObjectGroup(normalizedObjectGroupName.get(), objectGroupId)), Optional.empty());
        }
        if (normalizedCatalogName.isEmpty() || normalizedNamespaceName.isEmpty() || normalizedObjectName.isEmpty()) {
            throw new CatalogBadRequestException("Either object group name or object name must be provided");
        }
        UUID objectId = this.tableService.resolveIdForMaintenance(principal, normalizedCatalogName.get(), normalizedNamespaceName.get(), normalizedObjectName.get());
        return new MaintenanceOperationTarget(Optional.empty(), Optional.of(new MaintenanceOperationObject(normalizedCatalogName.get(), normalizedNamespaceName.get(), normalizedObjectName.get(), objectId)));
    }

    public void cancelOperation(AuthenticatedPrincipal principal, UUID operationId) {
        MaintenanceOperation operation = this.activeOperationsById.get(operationId);
        if (operation == null) {
            Optional<MaintenanceOperationInfoEx> info = this.operationStore.info(operationId);
            if (info.isEmpty()) {
                throw new CatalogMaintenanceOperationDoesNotExistException(operationId);
            }
            return;
        }
        UUID engineId = operation.engineId();
        Optional<ComputeEngineSession> maybeEngineSession = this.computeEngineService.createSession(engineId);
        if (maybeEngineSession.isEmpty()) {
            return;
        }
        try (ComputeEngineSession engineSession = maybeEngineSession.get();){
            this.authorizer.authorizeComputeEngineCancelOperation(principal, engineSession.toSecurable());
        }
        operation.cancel(String.format("Cancelled by \"%s\"", principal.name()));
    }

    public MaintenanceOperationInfo operationInfo(AuthenticatedPrincipal principal, UUID operationId) {
        MaintenanceOperationInfoEx info = this.operationStore.info(operationId).orElseThrow(() -> new CatalogMaintenanceOperationDoesNotExistException(operationId));
        this.authorizer.authorizeComputeEngineDescribe(principal, info.toEngineSecurable());
        return info.info();
    }

    public MaintenanceOperationListResponse listOperations(AuthenticatedPrincipal currentPrincipal, ResultPage page, Optional<Map<String, Object>> filter) {
        AuthorizerComputeEnginePredicate authorizerPredicate = this.authorizer.authorizeComputeEngineListOperations(currentPrincipal);
        return this.operationStore.list(page, info -> authorizerPredicate.test(info.toEngineSecurable()), filter.isPresent() ? QueryCondition.parseFilter(filter.get(), MaintenanceOperationStore.OPERATION_LIST_FILTER_FIELDS) : List.of());
    }

    public void createJob(AuthenticationContext authenticationContext, MaintenanceJobCreateRequest request) {
        AuthenticatedPrincipal currentPrincipal;
        String normalizedJobName = CatalogObjectNameValidation.VALIDATION_MAINTENANCE_JOB.normalizeObjectName(request.getJobName());
        MaintenanceOperationConfig operationConfig = request.getOperationConfig();
        if (operationConfig == null) {
            throw new CatalogBadRequestException("Operation config cannot be empty");
        }
        String operationName = CatalogUtils.normalizeNotEmpty(operationConfig.getOperationName(), "Operation name cannot be empty");
        String engineName = operationConfig.getEngineName() != null ? CatalogObjectNameValidation.VALIDATION_COMPUTE_ENGINE.normalizeObjectName(operationConfig.getEngineName()) : "local";
        JobScheduleConfig scheduleConfig = request.getScheduleConfig();
        if (scheduleConfig == null) {
            throw new CatalogBadRequestException("Schedule config cannot be empty");
        }
        ScheduleTimeResolver scheduleTimeResolver = this.scheduleTimeResolver(scheduleConfig);
        AuthenticatedPrincipal runAsPrincipal = currentPrincipal = authenticationContext.subject();
        if (request.getRunAs() != null && !request.getRunAs().equals(currentPrincipal.name())) {
            runAsPrincipal = this.runAsPrincipal(authenticationContext, request.getRunAs());
        }
        String normalizedObjectGroupName = CatalogObjectNameValidation.VALIDATION_OBJECT_GROUP.normalizeObjectName(request.getTargetObjectGroupName());
        UUID objectGroupId = this.objectGroupService.resolveIdForMaintenance(normalizedObjectGroupName);
        try (ComputeEngineSession session = this.computeEngineService.createSession(engineName);){
            this.authorizer.authorizeJobCreate(authenticationContext.subject(), session.toSecurable());
            ((CatalogComputeEngine)session.resource()).createOperationFactory(operationName, Optional.ofNullable(operationConfig.getParameters()).orElse(Map.of()));
            UUID jobId = this.jobStore.createNewJobIfNotExists(normalizedJobName, scheduleConfig.getScheduleExpression(), scheduleConfig.getTimeZoneId(), request.getDescription(), currentPrincipal.id(), runAsPrincipal.id(), operationName, operationConfig.getParameters(), engineName, session.engineId(), request.getOperationConfig().getEngineConfig(), normalizedObjectGroupName, objectGroupId).orElseThrow(() -> new CatalogMaintenanceJobAlreadyExistsException(normalizedJobName));
            this.startJobScheduling(new ActiveJob(jobId, scheduleTimeResolver));
        }
    }

    private void startJobScheduling(ActiveJob job) {
        this.scheduledJobsById.put(job.id, job);
        this.withJobLock(job.id, currentJob -> {
            if (currentJob != job) {
                return;
            }
            job.scheduleNextExecution();
        });
    }

    private void startScheduledOperation(ActiveJob job) {
        this.withJobLock(job.id, currentJob -> {
            if (currentJob != job) {
                return;
            }
            MaintenanceJobInfoEx info = this.jobStore.info(job.id, this.nextExecutionTime).orElse(null);
            if (info == null) {
                return;
            }
            ComputeEngineSession session = this.computeEngineService.createSession(info.engineId()).orElse(null);
            if (session == null) {
                return;
            }
            AuthenticatedPrincipal runAsPrincipal = null;
            try {
                AuthenticationContext authenticationContext = this.principalService.authenticate(info.ownerId()).orElseThrow(CatalogAuthenticationException::new).toContext();
                if (!info.runAsPrincipalId().equals(authenticationContext.subject().id())) {
                    authenticationContext = this.principalService.impersonate(authenticationContext, info.runAsPrincipalId());
                }
                runAsPrincipal = authenticationContext.subject();
                this.authorizer.authorizeComputeEngineStartOperation(runAsPrincipal, session.toSecurable());
                MaintenanceOperationConfig operationConfig = info.info().getOperationConfig();
                CatalogComputeEngineOperationFactory operationFactory = ((CatalogComputeEngine)session.resource()).createOperationFactory(operationConfig.getOperationName(), Optional.ofNullable(operationConfig.getParameters()).orElse(Map.of()));
                MaintenanceOperationTarget operationTarget = new MaintenanceOperationTarget(Optional.of(new MaintenanceOperationObjectGroup(info.info().getTargetObjectGroupName(), info.targetObjectGroupId())), Optional.empty());
                AuthenticatedPrincipal operationPrincipal = runAsPrincipal;
                job.lastOperationId = this.startOperationInternal(operationPrincipal, operationFactory, operationTarget, operationConfig, session, (ActiveJob)currentJob);
            }
            catch (Throwable e) {
                session.close();
                log.error(e, String.format("Failed to start scheduled operation for job \"%s\" with id %s", info.info().getJobName(), currentJob.id.toString()));
                this.operationStore.createFailed(e.getMessage(), info.engineId(), UUID.randomUUID(), info.info().getOperationConfig().getOperationName(), info.info().getOperationConfig().getParameters(), Optional.of(job.id), Optional.of(info.targetObjectGroupId()), Optional.empty(), info.info().getOperationConfig().getEngineConfig(), runAsPrincipal != null ? runAsPrincipal.id() : null, currentJob.executionTime.timestamp());
                currentJob.scheduleNextExecution();
            }
        });
    }

    private ScheduleExecutionTime nextExecutionTime(ScheduleTimeResolver scheduleTimeResolver, ScheduleExecutionTime previousTime) {
        long currentTime = this.currentTimeSupplier.currentTimeMillis();
        if (previousTime != null) {
            currentTime = Math.max(currentTime, previousTime.timestamp());
        }
        return scheduleTimeResolver.nextExecutionTime(currentTime);
    }

    public void updateJob(AuthenticationContext authenticationContext, String jobName, MaintenanceJobUpdateRequest request) {
        if (request.getNewJobName() == null && request.getDescription() == null && request.getTargetObjectGroupName() == null && request.getScheduleExpression() == null && request.getScheduleTimeZoneId() == null && request.getRunAs() == null) {
            throw new CatalogBadRequestException("Nothing to update");
        }
        AuthenticatedPrincipal currentPrincipal = authenticationContext.subject();
        String normalizedJobName = CatalogObjectNameValidation.VALIDATION_MAINTENANCE_JOB.normalizeObjectName(jobName);
        UUID jobId = this.jobStore.getJobId(normalizedJobName).orElseThrow(() -> new CatalogMaintenanceJobDoesNotExistException(normalizedJobName));
        this.withJobLock(jobId, job -> {
            String description;
            MaintenanceJobInfoEx info = this.jobStore.info(job.id, this.nextExecutionTime).orElseThrow(() -> new CatalogMaintenanceJobDoesNotExistException(jobName));
            this.authorizer.authorizeJobAlter(currentPrincipal, info.toSecurable());
            JobScheduleConfig scheduleConfig = info.info().getScheduleConfig();
            ScheduleTimeResolver newScheduleTimeResolver = null;
            if (request.getScheduleExpression() != null || request.getScheduleTimeZoneId() != null) {
                scheduleConfig = new JobScheduleConfig(request.getScheduleExpression() != null ? request.getScheduleExpression().getValue() : info.info().getScheduleConfig().getScheduleExpression(), request.getScheduleTimeZoneId() != null ? request.getScheduleTimeZoneId().getValue() : info.info().getScheduleConfig().getTimeZoneId());
                newScheduleTimeResolver = this.scheduleTimeResolver(scheduleConfig);
            }
            UUID objectGroupId = info.targetObjectGroupId();
            if (request.getTargetObjectGroupName() != null) {
                objectGroupId = this.objectGroupService.resolveIdForMaintenance(request.getTargetObjectGroupName().getValue());
            }
            UUID runAsPrincipalId = info.runAsPrincipalId();
            if (request.getRunAs() != null) {
                AuthenticatedPrincipal runAsPrincipal = this.runAsPrincipal(authenticationContext, request.getRunAs().getValue());
                runAsPrincipalId = runAsPrincipal.id();
            }
            String newJobName = request.getNewJobName() != null ? CatalogObjectNameValidation.VALIDATION_MAINTENANCE_JOB.normalizeObjectName(request.getNewJobName().getValue()) : info.info().getJobName();
            String string = description = request.getDescription() != null ? request.getDescription().getValue() : info.info().getDescription();
            if (newJobName.equals(info.info().getJobName()) && Objects.equals(description, info.info().getDescription()) && scheduleConfig.equals((Object)info.info().getScheduleConfig()) && objectGroupId.equals(info.targetObjectGroupId()) && runAsPrincipalId.equals(info.runAsPrincipalId())) {
                return;
            }
            if (!this.jobStore.updateIfExists(info.id(), newJobName, scheduleConfig.getScheduleExpression(), scheduleConfig.getTimeZoneId(), description, runAsPrincipalId, objectGroupId)) {
                throw new CatalogMaintenanceJobDoesNotExistException(info.info().getJobName());
            }
            if (newScheduleTimeResolver != null) {
                if (job.nextExecutionFuture != null) {
                    job.nextExecutionFuture.cancel(true);
                }
                ActiveJob newJob = new ActiveJob(info.id(), newScheduleTimeResolver);
                this.scheduledJobsById.put(newJob.id, newJob);
                newJob.scheduleNextExecution();
            }
        });
    }

    public boolean grantJobOwnership(AuthenticatedPrincipal currentPrincipal, Optional<String> expectedEngineName, String jobName, AuthenticatedPrincipal newOwnerPrincipal) {
        MaintenanceJobInfoEx info = this.jobInfo(jobName);
        this.authorizer.authorizeGrantOwnership(currentPrincipal, info.toSecurable());
        if (expectedEngineName.isPresent()) {
            String actualEngineName = info.info().getOperationConfig().getEngineName();
            if (!Objects.equals(expectedEngineName.get(), actualEngineName)) {
                throw new CatalogBadRequestException(String.format("Job \"%s\" exists, but doesn't use the \"%s\" compute engine", jobName, expectedEngineName.get()));
            }
        }
        if (info.ownerId().equals(newOwnerPrincipal.id())) {
            return false;
        }
        if (!this.jobStore.updateOwnerIfExists(info.id(), newOwnerPrincipal.id())) {
            throw new CatalogMaintenanceJobDoesNotExistException(info.info().getJobName());
        }
        return true;
    }

    public void deleteJob(AuthenticatedPrincipal currentPrincipal, String jobName) {
        MaintenanceJobInfoEx info = this.jobInfo(jobName);
        this.authorizer.authorizeJobDrop(currentPrincipal, info.toSecurable());
        this.withJobLock(info.id(), job -> {
            MaintenanceOperation operation;
            this.scheduledJobsById.remove(info.id());
            if (!this.jobStore.deleteJob(info.id())) {
                throw new CatalogMaintenanceJobDoesNotExistException(jobName);
            }
            if (job.nextExecutionFuture != null) {
                job.nextExecutionFuture.cancel(true);
            }
            if (job.lastOperationId != null && (operation = this.activeOperationsById.get(job.lastOperationId)) != null) {
                operation.cancel("Job has been deleted");
            }
        });
        this.privilegeInternalService.clearPrivileges();
    }

    public MaintenanceJobInfo getJob(AuthenticatedPrincipal currentPrincipal, String jobName) {
        MaintenanceJobInfoEx info = this.jobInfo(jobName);
        this.authorizer.authorizeJobDescribe(currentPrincipal, info.toSecurable());
        return info.info();
    }

    public MaintenanceJobListResponse listJobs(AuthenticatedPrincipal currentPrincipal, ResultPage page, Optional<Map<String, Object>> filter) {
        AuthorizerJobPredicate predicate = this.authorizer.authorizeJobList(currentPrincipal);
        PageData<MaintenanceJobInfoEx> pageData = this.jobStore.list(page, info -> predicate.test(info.toSecurable()), filter.isPresent() ? QueryCondition.parseFilter(filter.get(), MaintenanceJobStore.JOB_LIST_FILTER_FIELDS) : List.of(), this.nextExecutionTime);
        return new MaintenanceJobListResponse(pageData.pageData().stream().map(MaintenanceJobInfoEx::info).toList(), pageData.nextPageToken());
    }

    @VisibleForTesting
    public MaintenanceJobInfoEx jobInfo(String jobName) {
        String normalizedJobName = CatalogObjectNameValidation.VALIDATION_MAINTENANCE_JOB.normalizeObjectName(jobName);
        Optional<MaintenanceJobInfoEx> info = this.jobStore.info(normalizedJobName, this.nextExecutionTime);
        return info.orElseThrow(() -> new CatalogMaintenanceJobDoesNotExistException(normalizedJobName));
    }

    private AuthenticatedPrincipal runAsPrincipal(AuthenticationContext authenticationContext, String runAs) {
        try {
            return this.principalService.impersonate(authenticationContext, runAs).subject();
        }
        catch (CatalogAuthenticationException e) {
            throw new CatalogBadRequestException(String.format("Failed to authenticate the run-as user: \"%s\"", runAs));
        }
    }

    private ScheduleTimeResolver scheduleTimeResolver(JobScheduleConfig scheduleConfig) {
        ZoneId zoneId;
        String scheduleExpression = scheduleConfig.getScheduleExpression();
        if (scheduleExpression == null || scheduleExpression.isEmpty()) {
            throw new CatalogBadRequestException("Schedule expression cannot be empty");
        }
        try {
            zoneId = scheduleConfig.getTimeZoneId() == null ? UTC_TIME_ZONE : ZoneId.of(scheduleConfig.getTimeZoneId());
        }
        catch (DateTimeException e) {
            throw new CatalogBadRequestException("Invalid timezone ID");
        }
        try {
            return ScheduleTimeResolver.createCronResolver(scheduleExpression, zoneId);
        }
        catch (IllegalArgumentException e) {
            throw new CatalogBadRequestException(String.format("Invalid cron schedule: %s", e.getMessage()));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void withJobLock(UUID jobId, Consumer<ActiveJob> action) {
        ActiveJob job = this.scheduledJobsById.get(jobId);
        if (job == null) {
            return;
        }
        job.lock.lock();
        try {
            action.accept(job);
        }
        finally {
            job.lock.unlock();
        }
    }

    class ActiveJob {
        final Lock lock;
        final UUID id;
        ScheduleTimeResolver scheduleTimeResolver;
        UUID lastOperationId;
        ScheduleExecutionTime executionTime;
        ScheduledFuture<?> nextExecutionFuture;

        ActiveJob(UUID id, ScheduleTimeResolver scheduleTimeResolver) {
            this.id = id;
            this.scheduleTimeResolver = scheduleTimeResolver;
            this.lock = (Lock)MaintenanceOperationService.this.jobLock.get((Object)id);
        }

        void scheduleNextExecution() {
            this.executionTime = MaintenanceOperationService.this.nextExecutionTime(this.scheduleTimeResolver, this.executionTime);
            this.nextExecutionFuture = MaintenanceOperationService.this.scheduledExecutor.schedule(() -> MaintenanceOperationService.this.startScheduledOperation(this), this.executionTime.timeToNextExecution(), TimeUnit.MILLISECONDS);
        }
    }
}

