/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.iceberg.rest;

import com.google.inject.Inject;
import io.airlift.log.Logger;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwe;
import io.jsonwebtoken.Jwt;
import io.jsonwebtoken.JwtParser;
import io.jsonwebtoken.impl.DefaultJwtParserBuilder;
import io.jsonwebtoken.io.Deserializer;
import io.jsonwebtoken.jackson.io.JacksonDeserializer;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.FormParam;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.Request;
import jakarta.ws.rs.core.Response;
import java.util.Objects;
import java.util.Optional;
import org.apache.iceberg.rest.responses.OAuthTokenResponse;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.core.principal.AuthenticationContext;
import ru.cedrusdata.catalog.core.principal.JwtCredentials;
import ru.cedrusdata.catalog.core.principal.PrincipalService;
import ru.cedrusdata.catalog.iceberg.oauth.OAuthError;
import ru.cedrusdata.catalog.iceberg.oauth.OAuthErrorType;
import ru.cedrusdata.catalog.server.security.ResourceSecurity;
import ru.cedrusdata.catalog.server.security.ResourceType;
import ru.cedrusdata.catalog.spi.exception.CatalogAuthenticationException;
import ru.cedrusdata.catalog.spi.exception.CatalogAuthorizationException;
import ru.cedrusdata.catalog.spi.exception.CatalogPrincipalDoesNotExistException;
import ru.cedrusdata.catalog.spi.model.AccessTokenCreateTemporaryResponse;

@ResourceSecurity(value=ResourceSecurity.Type.PUBLIC)
@ResourceType(value=ResourceType.Type.ICEBERG)
@Path(value="/catalog/iceberg/v1")
public class IcebergOAuthResource {
    private static final Logger logger = Logger.get(IcebergOAuthResource.class);
    private static final String TOKEN_TYPE_BEARER = "bearer";
    private static final String GRANT_TYPE_CLIENT_CREDENTIALS = "client_credentials";
    private static final String GRANT_TYPE_TOKEN_EXCHANGE = "urn:ietf:params:oauth:grant-type:token-exchange";
    private final PrincipalService principalService;
    private final JwtParser icebergExchangeTokenJwtParser;

    @Inject
    public IcebergOAuthResource(PrincipalService principalService) {
        this.principalService = Objects.requireNonNull(principalService, "userService");
        this.icebergExchangeTokenJwtParser = new DefaultJwtParserBuilder().unsecured().json((Deserializer)new JacksonDeserializer()).build();
    }

    @POST
    @Path(value="/oauth/tokens")
    @Consumes(value={"application/x-www-form-urlencoded"})
    @Produces(value={"application/json"})
    public Response getToken(Request request, @FormParam(value="grant_type") String grantType, @FormParam(value="scope") String scope, @FormParam(value="client_id") String clientId, @FormParam(value="client_secret") String clientSecret, @FormParam(value="requested_token_type") String requestedTokenType, @FormParam(value="subject_token") String subjectToken, @FormParam(value="subject_token_type") String subjectTokenType, @FormParam(value="actor_token") String actorToken, @FormParam(value="actor_token_type") String actorTokenType) {
        if (requestedTokenType != null) {
            return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_REQUEST, String.format("Invalid requested token type \"%s\", expected empty", requestedTokenType));
        }
        return switch (grantType) {
            case GRANT_TYPE_CLIENT_CREDENTIALS -> this.processClientCredentials(scope, clientId, clientSecret);
            case GRANT_TYPE_TOKEN_EXCHANGE -> this.processTokenExchange(scope, subjectToken, subjectTokenType, actorToken, actorTokenType);
            default -> IcebergOAuthResource.errorResponse(OAuthErrorType.UNSUPPORTED_GRANT_TYPE, "Unsupported grant type: " + grantType);
        };
    }

    private Response processClientCredentials(String scope, String clientId, String clientSecret) {
        if (clientId == null || clientId.isEmpty()) {
            return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_REQUEST, "Client ID cannot be empty");
        }
        if (clientSecret == null || clientSecret.isEmpty()) {
            return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_REQUEST, "Client secret cannot be empty");
        }
        if (!"catalog".equals(scope)) {
            return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_SCOPE, String.format("Invalid scope \"%s\", expected \"%s\"", scope, "catalog"));
        }
        Optional<AuthenticatedPrincipal> principal = this.principalService.authenticateAccessToken(clientId + ":" + clientSecret);
        if (principal.isEmpty()) {
            return IcebergOAuthResource.errorResponseUnauthorized();
        }
        return this.createJwtTokenForSubject(principal.get().toContext(), principal.get().name());
    }

    private Response processTokenExchange(String scope, String subjectToken, String subjectTokenType, String actorToken, String actorTokenType) {
        JwtCredentials actorJwtCredentials;
        String subject;
        if (!"catalog".equals(scope)) {
            return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_SCOPE, String.format("Invalid scope \"%s\", expected \"%s\"", scope, "catalog"));
        }
        if (subjectToken == null || subjectToken.trim().isEmpty()) {
            return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_REQUEST, "Subject token cannot be empty");
        }
        if (actorTokenType != null) {
            if (!"urn:ietf:params:oauth:token-type:jwt".equals(subjectTokenType)) {
                return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_REQUEST, String.format("Invalid subject token type \"%s\", expected \"%s\"", subjectTokenType, "urn:ietf:params:oauth:token-type:jwt"));
            }
            Optional<String> maybeSubject = this.unwrapIcebergTokenExchangeSubject(subjectToken);
            if (maybeSubject.isEmpty()) {
                return IcebergOAuthResource.errorResponseUnauthorized();
            }
            subject = maybeSubject.get();
            if (!"urn:ietf:params:oauth:token-type:access_token".equals(actorTokenType)) {
                return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_REQUEST, String.format("Invalid actor token type \"%s\", expected \"%s\"", actorTokenType, "urn:ietf:params:oauth:token-type:access_token"));
            }
            if (actorToken == null || actorToken.trim().isEmpty()) {
                return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_REQUEST, "Actor token cannot be empty");
            }
            Optional<JwtCredentials> maybeActorJwtCredentials = this.principalService.getTemporaryAccessTokenCredentials(actorToken);
            if (maybeActorJwtCredentials.isEmpty()) {
                return IcebergOAuthResource.errorResponseUnauthorized();
            }
            actorJwtCredentials = maybeActorJwtCredentials.get();
        } else {
            if (!"urn:ietf:params:oauth:token-type:access_token".equals(subjectTokenType)) {
                return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_REQUEST, String.format("Invalid subject token type \"%s\", expected \"%s\"", subjectTokenType, "urn:ietf:params:oauth:token-type:access_token"));
            }
            Optional<JwtCredentials> maybeSubjectJwtCredentials = this.principalService.getTemporaryAccessTokenCredentials(subjectToken);
            if (maybeSubjectJwtCredentials.isEmpty()) {
                return IcebergOAuthResource.errorResponseUnauthorized();
            }
            actorJwtCredentials = maybeSubjectJwtCredentials.get();
            subject = actorJwtCredentials.subject();
        }
        return this.createJwtTokenForSubject(actorJwtCredentials, subject);
    }

    private Response createJwtTokenForSubject(JwtCredentials actorJwtCredentials, String subject) {
        Optional<AuthenticatedPrincipal> currentActorPrincipal = this.principalService.authenticate(actorJwtCredentials.actor());
        if (currentActorPrincipal.isEmpty()) {
            return IcebergOAuthResource.errorResponseUnauthorized();
        }
        if (subject.equalsIgnoreCase(actorJwtCredentials.actor()) && subject.equalsIgnoreCase(actorJwtCredentials.subject())) {
            return this.createJwtTokenForSubject(new AuthenticationContext(currentActorPrincipal.get(), currentActorPrincipal.get()), actorJwtCredentials.actor());
        }
        if (actorJwtCredentials.actor().equalsIgnoreCase(actorJwtCredentials.subject()) || actorJwtCredentials.subject().equalsIgnoreCase(subject)) {
            Optional<AuthenticatedPrincipal> currentSubjectPrincipal = this.principalService.authenticate(actorJwtCredentials.subject());
            if (currentSubjectPrincipal.isEmpty()) {
                return IcebergOAuthResource.errorResponseUnauthorized();
            }
            return this.createJwtTokenForSubject(new AuthenticationContext(currentActorPrincipal.get(), currentSubjectPrincipal.get()), subject);
        }
        return IcebergOAuthResource.errorResponseUnauthorized();
    }

    private Response createJwtTokenForSubject(AuthenticationContext actorContext, String subject) {
        try {
            AccessTokenCreateTemporaryResponse response = this.principalService.createTemporaryAccessToken(actorContext, subject);
            OAuthTokenResponse tokenResponse = OAuthTokenResponse.builder().withToken(response.getAccessToken()).withTokenType(TOKEN_TYPE_BEARER).withIssuedTokenType("urn:ietf:params:oauth:token-type:access_token").setExpirationInSeconds(response.getTtl().intValue()).build();
            return Response.ok((Object)tokenResponse).build();
        }
        catch (CatalogAuthenticationException | CatalogAuthorizationException | CatalogPrincipalDoesNotExistException e) {
            return IcebergOAuthResource.errorResponseUnauthorized();
        }
        catch (Exception e) {
            return IcebergOAuthResource.errorResponse(OAuthErrorType.INVALID_REQUEST, e.getMessage());
        }
    }

    private static Response errorResponse(OAuthErrorType errorType, String message) {
        return Response.status((int)errorType.getHttpCode()).entity((Object)new OAuthError(errorType, message)).build();
    }

    private static Response errorResponseUnauthorized() {
        return IcebergOAuthResource.errorResponse(OAuthErrorType.UNAUTHORIZED_CLIENT, "Authentication failed");
    }

    private Optional<String> unwrapIcebergTokenExchangeSubject(String subjectJwtToken) {
        Jwt jwt;
        try {
            jwt = this.icebergExchangeTokenJwtParser.parse((CharSequence)subjectJwtToken);
        }
        catch (Exception e) {
            logger.warn((Throwable)e, "Failed to parse JWT token for Iceberg token exchange: " + e.getMessage());
            return Optional.empty();
        }
        try {
            String subject = ((Claims)((Jwt)jwt.accept(Jwe.UNSECURED_CLAIMS)).getPayload()).getSubject();
            return Optional.ofNullable(subject);
        }
        catch (Exception e) {
            logger.warn((Throwable)e, "Failed to get subject from JWT token for Iceberg token exchange: " + e.getMessage());
            return Optional.empty();
        }
    }
}

