/*
 * Decompiled with CFR 0.152.
 */
package ru.cedrusdata.catalog.core.security.provider;

import com.google.common.annotations.VisibleForTesting;
import com.google.inject.Inject;
import io.airlift.log.Logger;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import ru.cedrusdata.catalog.CatalogObjectNameValidation;
import ru.cedrusdata.catalog.CatalogUtils;
import ru.cedrusdata.catalog.core.PageData;
import ru.cedrusdata.catalog.core.ResultPageIterator;
import ru.cedrusdata.catalog.core.principal.AuthenticatedPrincipal;
import ru.cedrusdata.catalog.core.security.authorization.Authorizer;
import ru.cedrusdata.catalog.core.security.authorization.PrivilegeInternalService;
import ru.cedrusdata.catalog.core.security.authorization.predicate.AuthorizerSecurityProviderPredicate;
import ru.cedrusdata.catalog.core.security.authorization.securable.SecurityProviderInternalSecurable;
import ru.cedrusdata.catalog.core.security.provider.FaultySecurityProvider;
import ru.cedrusdata.catalog.core.security.provider.SecurityProviderDetails;
import ru.cedrusdata.catalog.core.security.provider.SecurityProviderSession;
import ru.cedrusdata.catalog.core.security.provider.UsageAwareSecurityProvider;
import ru.cedrusdata.catalog.plugin.CatalogPluginContextProvider;
import ru.cedrusdata.catalog.plugin.CatalogPluginRegistry;
import ru.cedrusdata.catalog.plugin.PluginPropertyClassifier;
import ru.cedrusdata.catalog.spi.client.ResultPage;
import ru.cedrusdata.catalog.spi.exception.CatalogBadRequestException;
import ru.cedrusdata.catalog.spi.exception.CatalogException;
import ru.cedrusdata.catalog.spi.exception.CatalogInternalServerErrorException;
import ru.cedrusdata.catalog.spi.exception.CatalogSecurityProviderAlreadyExistsException;
import ru.cedrusdata.catalog.spi.exception.CatalogSecurityProviderCheckException;
import ru.cedrusdata.catalog.spi.exception.CatalogSecurityProviderDoesNotExistException;
import ru.cedrusdata.catalog.spi.model.SecurityProviderCheckResponse;
import ru.cedrusdata.catalog.spi.model.SecurityProviderCreateRequest;
import ru.cedrusdata.catalog.spi.model.SecurityProviderInfo;
import ru.cedrusdata.catalog.spi.model.SecurityProviderListResponse;
import ru.cedrusdata.catalog.spi.model.SecurityProviderUpdateRequest;
import ru.cedrusdata.catalog.spi.model.UpdateResponse;
import ru.cedrusdata.catalog.spi.security.CatalogSecurityProvider;
import ru.cedrusdata.catalog.spi.security.CatalogSecurityProviderProvider;
import ru.cedrusdata.catalog.store.SecurityProviderStore;

public class SecurityProviderService {
    private static final Logger log = Logger.get(SecurityProviderService.class);
    public static final String INTERNAL_PROVIDER_NAME = "local";
    public static final UUID INTERNAL_PROVIDER_ID = UUID.fromString("0d1a8124-c1d9-416c-a05e-1e56085e6e16");
    private final Authorizer authorizer;
    private final SecurityProviderStore store;
    private final CatalogPluginRegistry pluginRegistry;
    private final CatalogPluginContextProvider pluginContextProvider;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final Lock lock = new ReentrantLock();
    private final ConcurrentHashMap<UUID, UsageAwareSecurityProvider> providersById = new ConcurrentHashMap();
    private final PrivilegeInternalService privilegeInternalService;

    @Inject
    public SecurityProviderService(Authorizer authorizer, SecurityProviderStore store, CatalogPluginRegistry pluginRegistry, CatalogPluginContextProvider pluginContextProvider, PrivilegeInternalService privilegeInternalService) {
        this.authorizer = authorizer;
        this.store = store;
        this.pluginRegistry = pluginRegistry;
        this.pluginContextProvider = pluginContextProvider;
        this.privilegeInternalService = privilegeInternalService;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @PostConstruct
    public void initialize() {
        if (!this.initialized.compareAndSet(false, true)) {
            return;
        }
        this.lock.lock();
        try {
            ResultPageIterator iterator = new ResultPageIterator(page -> this.store.list((ResultPage)page, provider -> true), 100);
            while (iterator.hasNext()) {
                SecurityProviderDetails details = (SecurityProviderDetails)iterator.next();
                if (INTERNAL_PROVIDER_NAME.equals(details.name())) continue;
                Optional<CatalogSecurityProviderProvider> provider = this.pluginRegistry.getCatalogSecurityProviderProvider(details.type());
                if (provider.isEmpty()) {
                    throw new CatalogInternalServerErrorException(String.format("No provider for %s security provider \"%s\"", details.type(), details.name()));
                }
                UsageAwareSecurityProvider securityProvider = this.createProvider(details.id(), details.name(), details.type(), details.properties(), details.ownerId(), true);
                ((CatalogSecurityProvider)securityProvider.resource()).jmxRegister();
                this.providersById.put(details.id(), securityProvider);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @PreDestroy
    public void stop() {
        this.lock.lock();
        try {
            for (UsageAwareSecurityProvider provider : this.providersById.values()) {
                provider.close();
            }
            this.providersById.clear();
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void createProvider(AuthenticatedPrincipal principal, SecurityProviderCreateRequest request) {
        this.authorizer.authorizeSecurityProviderCreate(principal);
        String name = CatalogObjectNameValidation.VALIDATION_SECURITY_PROVIDER.normalizeObjectName(request.getSecurityProviderName());
        if (INTERNAL_PROVIDER_NAME.equals(name)) {
            throw new CatalogBadRequestException(String.format("Reserved provider name \"%s\" cannot be used for custom provider", name));
        }
        String type = CatalogUtils.normalizeNotEmpty(request.getType(), "Security provider type cannot be empty");
        if (this.pluginRegistry.getCatalogSecurityProviderProvider(type).isEmpty()) {
            throw new CatalogBadRequestException(String.format("Unsupported security provider type \"%s\"", type));
        }
        UUID id = UUID.randomUUID();
        this.lock.lock();
        try {
            UsageAwareSecurityProvider newProvider = this.createProvider(id, name, type, request.getProperties(), Optional.of(principal.id()), false);
            if (!this.updateStoreAndProviderMap(id, newProvider, () -> this.store.createIfNotExists(id, name, request.getDescription(), type, request.getProperties(), principal.id()))) {
                throw new CatalogSecurityProviderAlreadyExistsException(name);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public SecurityProviderCheckResponse checkProvider(AuthenticatedPrincipal principal, String providerName) {
        String normalizedName = CatalogObjectNameValidation.VALIDATION_SECURITY_PROVIDER.normalizeObjectName(providerName);
        if (INTERNAL_PROVIDER_NAME.equals(normalizedName)) {
            return new SecurityProviderCheckResponse(Boolean.valueOf(true));
        }
        try (SecurityProviderSession session = this.createSession(normalizedName);){
            this.authorizer.authorizeSecurityProviderCheck(principal, session.toSecurable());
            try {
                ((CatalogSecurityProvider)session.resource()).check();
                SecurityProviderCheckResponse securityProviderCheckResponse = new SecurityProviderCheckResponse(Boolean.valueOf(true));
                return securityProviderCheckResponse;
            }
            catch (Exception e) {
                throw new CatalogSecurityProviderCheckException(normalizedName, e.toString());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public UpdateResponse updateProvider(AuthenticatedPrincipal principal, String providerName, SecurityProviderUpdateRequest request) {
        if (request.getNewProviderName() == null && request.getDescription() == null && request.getUpdatedProperties() == null && request.getRemovedProperties() == null) {
            throw new CatalogBadRequestException("Nothing to update");
        }
        if (INTERNAL_PROVIDER_NAME.equals(providerName = CatalogObjectNameValidation.VALIDATION_SECURITY_PROVIDER.normalizeObjectName(providerName))) {
            throw new CatalogBadRequestException(String.format("Security provider \"%s\" cannot be updated", providerName));
        }
        this.lock.lock();
        try {
            SecurityProviderDetails currentDetails = this.resolveDetails(providerName);
            this.authorizer.authorizeSecurityProviderAlter(principal, currentDetails.toSecurable());
            UUID id = currentDetails.id();
            String newName = request.getNewProviderName() != null ? CatalogObjectNameValidation.VALIDATION_SECURITY_PROVIDER.normalizeObjectName(request.getNewProviderName().getValue()) : providerName;
            String newDescription = request.getDescription() != null ? request.getDescription().getValue() : currentDetails.description();
            Map<String, String> newProperties = CatalogUtils.mergeProperties(currentDetails.properties(), request.getUpdatedProperties(), request.getRemovedProperties());
            if (currentDetails.name().equals(newName) && Objects.equals(currentDetails.description(), newDescription) && Objects.equals(currentDetails.properties(), newProperties)) {
                UpdateResponse updateResponse = new UpdateResponse(Boolean.valueOf(false));
                return updateResponse;
            }
            if (INTERNAL_PROVIDER_NAME.equals(newName)) {
                throw new CatalogBadRequestException(String.format("Reserved provider name \"%s\" cannot be used for custom provider", newName));
            }
            UsageAwareSecurityProvider newProvider = this.createProvider(id, newName, currentDetails.type(), newProperties, currentDetails.ownerId(), false);
            if (!this.updateStoreAndProviderMap(id, newProvider, () -> this.store.updateIfExists(id, newName, newDescription, newProperties))) {
                throw new CatalogSecurityProviderDoesNotExistException(providerName);
            }
            UpdateResponse updateResponse = new UpdateResponse(Boolean.valueOf(true));
            return updateResponse;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteProvider(AuthenticatedPrincipal principal, String providerName) {
        UsageAwareSecurityProvider removedProvider;
        String normalizedName = CatalogObjectNameValidation.VALIDATION_SECURITY_PROVIDER.normalizeObjectName(providerName);
        if (INTERNAL_PROVIDER_NAME.equals(normalizedName)) {
            throw new CatalogBadRequestException(String.format("Security provider \"%s\" cannot be deleted", normalizedName));
        }
        this.lock.lock();
        try {
            SecurityProviderDetails details = this.resolveDetails(providerName);
            this.authorizer.authorizeSecurityProviderDrop(principal, details.toSecurable());
            if (!this.store.deleteIfExists(details.id(), details.name())) {
                throw new CatalogSecurityProviderDoesNotExistException(normalizedName);
            }
            removedProvider = this.providersById.remove(details.id());
        }
        finally {
            this.lock.unlock();
        }
        if (removedProvider != null) {
            removedProvider.close();
        }
        this.privilegeInternalService.clearPrivileges();
    }

    public SecurityProviderInfo getProvider(AuthenticatedPrincipal principal, String providerName) {
        providerName = CatalogObjectNameValidation.VALIDATION_SECURITY_PROVIDER.normalizeObjectName(providerName);
        SecurityProviderDetails details = this.resolveDetails(providerName);
        this.authorizer.authorizeSecurityProviderDescribe(principal, details.toSecurable());
        return details.toInfo(this.pluginRegistry.securityProviderPropertyClassifier());
    }

    public SecurityProviderListResponse listProviders(AuthenticatedPrincipal principal, ResultPage page) {
        AuthorizerSecurityProviderPredicate predicate = this.authorizer.authorizeSecurityProviderList(principal);
        PluginPropertyClassifier propertyClassifier = this.pluginRegistry.securityProviderPropertyClassifier();
        PageData<SecurityProviderDetails> pageData = this.store.list(page, info -> predicate.test(info.toSecurable()));
        return new SecurityProviderListResponse(pageData.pageData().stream().map(details -> details.toInfo(propertyClassifier)).toList(), pageData.nextPageToken());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean grantOwnership(AuthenticatedPrincipal currentPrincipal, String providerName, AuthenticatedPrincipal newOwnerPrincipal) {
        String normalizedName = CatalogObjectNameValidation.VALIDATION_SECURITY_PROVIDER.normalizeObjectName(providerName);
        this.lock.lock();
        try {
            SecurityProviderDetails details = this.resolveDetails(normalizedName);
            this.authorizer.authorizeGrantOwnership(currentPrincipal, details.toSecurable());
            if (Objects.equals(details.ownerId().orElse(null), newOwnerPrincipal.id())) {
                boolean bl = false;
                return bl;
            }
            if (INTERNAL_PROVIDER_NAME.equals(normalizedName)) {
                boolean bl = this.store.updateOwner(details.id(), newOwnerPrincipal.id());
                return bl;
            }
            UsageAwareSecurityProvider newProvider = this.createProvider(details.id(), details.name(), details.type(), details.properties(), Optional.of(newOwnerPrincipal.id()), false);
            if (!this.updateStoreAndProviderMap(details.id(), newProvider, () -> this.store.updateOwner(details.id(), newOwnerPrincipal.id()))) {
                throw new CatalogSecurityProviderDoesNotExistException(normalizedName);
            }
            boolean bl = true;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    public UUID resolveProviderId(AuthenticatedPrincipal currentPrincipal, Optional<String> providerName) {
        if (providerName.isEmpty()) {
            return INTERNAL_PROVIDER_ID;
        }
        if (INTERNAL_PROVIDER_NAME.equals(providerName.get()) || providerName.get().isEmpty()) {
            return INTERNAL_PROVIDER_ID;
        }
        SecurityProviderInternalSecurable pluginSecurable = this.privilegeInternalService.bindSecurityProvider(CatalogObjectNameValidation.VALIDATION_SECURITY_PROVIDER.normalizeObjectName(providerName.get()));
        this.authorizer.authorizeSecurityProviderUse(currentPrincipal, pluginSecurable);
        return pluginSecurable.id();
    }

    public boolean useCustomAuthentication(UUID providerId) {
        return !INTERNAL_PROVIDER_ID.equals(providerId);
    }

    public boolean authenticate(UUID providerId, String user, Optional<String> password) {
        boolean bl;
        UsageAwareSecurityProvider provider = this.providersById.get(providerId);
        if (provider == null) {
            return false;
        }
        SecurityProviderSession session = new SecurityProviderSession(provider);
        try {
            bl = ((CatalogSecurityProvider)session.resource()).passwordAuthenticator().authenticate(user, password);
        }
        catch (Throwable throwable) {
            try {
                try {
                    session.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (CatalogSecurityProviderDoesNotExistException e) {
                return false;
            }
        }
        session.close();
        return bl;
    }

    @VisibleForTesting
    public SecurityProviderDetails resolveDetails(String providerName) {
        return this.store.details(providerName).orElseThrow(() -> new CatalogSecurityProviderDoesNotExistException(providerName));
    }

    private UsageAwareSecurityProvider createProvider(UUID providerId, String name, String type, Map<String, String> properties, Optional<UUID> ownerId, boolean startup) {
        CatalogSecurityProvider securityProvider;
        CatalogSecurityProviderProvider provider = this.pluginRegistry.getCatalogSecurityProviderProvider(type).orElseThrow(() -> new CatalogInternalServerErrorException(String.format("No provider for %s security provider \"%s\"", type, name)));
        try {
            securityProvider = provider.createSecurityProvider(this.pluginContextProvider.get(), name, properties);
        }
        catch (Exception e) {
            Object errorMessage = e instanceof CatalogBadRequestException ? ((CatalogBadRequestException)((Object)e)).getOriginalMessage() : String.format("%s security provider \"%s\" failed to start: %s", type, name, e.getMessage());
            if (startup) {
                errorMessage = (String)errorMessage + ". Please ask administrator to re-create the security provider.";
                CatalogInternalServerErrorException exception = new CatalogInternalServerErrorException((String)errorMessage, (Throwable)e);
                log.error((Throwable)e, (String)errorMessage);
                securityProvider = new FaultySecurityProvider((CatalogException)exception);
            }
            throw new CatalogBadRequestException((String)errorMessage, (Throwable)e);
        }
        return new UsageAwareSecurityProvider(securityProvider, name, providerId, ownerId);
    }

    private boolean updateStoreAndProviderMap(UUID id, UsageAwareSecurityProvider newProvider, Supplier<Boolean> update) {
        try {
            boolean storeUpdated = update.get();
            if (!storeUpdated) {
                newProvider.close();
            } else {
                UsageAwareSecurityProvider old = this.providersById.put(id, newProvider);
                if (old != null) {
                    old.close();
                }
                ((CatalogSecurityProvider)newProvider.resource()).jmxRegister();
            }
            return storeUpdated;
        }
        catch (Exception e) {
            newProvider.close();
            throw e;
        }
    }

    @VisibleForTesting
    SecurityProviderSession createSession(String providerName) {
        UsageAwareSecurityProvider provider = this.providerByName(providerName);
        if (provider == null) {
            throw new CatalogSecurityProviderDoesNotExistException(providerName);
        }
        return new SecurityProviderSession(provider);
    }

    private UsageAwareSecurityProvider providerByName(String providerName) {
        for (UsageAwareSecurityProvider provider : this.providersById.values()) {
            if (!providerName.equals(provider.name())) continue;
            return provider;
        }
        return null;
    }
}

